#include <Wire.h>

// A Battery reconditioner for the Arduino
// Written by Michael Suzuki

#define MAX_BATTERIES        4	// C++ is 0 based 0,1,2,3
const float aRef=3.13;			//Aref

// Timing 
unsigned long StartTime;		//Log Interval counter
unsigned long LogInterval;
unsigned long SampInterval=120.0; 	//Number of seconds in the LogInterval

// Analog pins
unsigned int PinV[MAX_BATTERIES];

//Digital output
int FET1[MAX_BATTERIES];
int FET2[MAX_BATTERIES];
int GreenLED;
int TypeSelPin;
	
unsigned int batteryNum;
byte j=0;

// Battery arrays
float StartBattV[MAX_BATTERIES];	// Start Battery voltage 
float LoadBattV[MAX_BATTERIES];		// Load Batt 
float IntRes[MAX_BATTERIES];		// Internal Resistance Batt 1
boolean finished;					// Check if finished
void error(char *str)
{
  Serial.print("error: ");
  Serial.println(str);
  while(1);
}

//Prototypes for utility functions
void printData();
float getBatteryVoltage(unsigned int batteryNum);
float getIntRes(float StartV, float LoadV);

void setup(void)
{
	// Assign analog pin numbers
	PinV[0]=5;		//Battery voltage 1
	PinV[1]=4;		//Battery voltage 2
	PinV[2]=3;		//Battery voltage 3
	PinV[3]=2;		//Battery voltage 4
	
	//Assign digital output
	FET1[0]=2;		//Discharge Bat1 1
	FET2[0]=8;		//Discharge Bat1 2
	FET1[1]=6;		//Discharge Bat2 1
	FET2[1]=7;		//Discharge Bat2 2
	FET1[2]=9;		//Discharge Bat3 1
	FET2[2]=10;		//Discharge Bat3 2
	FET1[3]=12;		//Discharge Bat4 1
	FET2[3]=13;		//Discharge Bat4 2
	GreenLED=11;
	TypeSelPin=3;	// If Pin is high, then high current device
					// If Pin is low (switch on), then low current device
	
	analogReference(EXTERNAL); //3.3V connected to AREF thru 3.3K AREF=3VDC
	//discard first analogReads
	for (j=0; j <= 5; j++){
		int dummy=analogRead(j);
	}
	// Set Digital pins	
	for(batteryNum= 0 ; batteryNum < MAX_BATTERIES ; batteryNum++)
	{
		pinMode(PinV[batteryNum],INPUT);
		pinMode(FET1[batteryNum], OUTPUT);
		pinMode(FET2[batteryNum], OUTPUT);
	}
	pinMode(GreenLED, OUTPUT);
	pinMode(TypeSelPin, INPUT);      // set TypeSel to input
	digitalWrite(TypeSelPin, HIGH);  // turn on 20K pullup resistor
	
  Serial.begin(9600);
  Serial.println();
  Serial.println("StartV1, LoadV1, IntRes1, StartV2, LoadV2, IntRes2, StartV3, LoadV3, IntRes3, StartV4, LoadV4, IntRes4");

  LogInterval=SampInterval*1000.0;	 //  LogInterval needs to be in milliseconds
  StartTime=millis(); //Initialize StartTime
}

void loop(void)
{
	// Get starting voltages
	for(batteryNum= 0 ; batteryNum < MAX_BATTERIES ; batteryNum++)
	{
		StartBattV[batteryNum] = getBatteryVoltage(PinV[batteryNum]);
	}

	// Activate FET1 to get Internal Resistance
	for(batteryNum= 0 ; batteryNum < MAX_BATTERIES ; batteryNum++)
	{
		digitalWrite(FET1[batteryNum], HIGH);
	}
	delay(200);  
	for(batteryNum= 0 ; batteryNum < MAX_BATTERIES ; batteryNum++)
	{
		LoadBattV[batteryNum] = getBatteryVoltage(PinV[batteryNum]);
		digitalWrite(FET1[batteryNum], LOW);
		IntRes[batteryNum]=getIntRes(StartBattV[batteryNum], LoadBattV[batteryNum]);
	}
	printData();
	
	//Check for  any batteries above 0.4 Volts
	finished=true;
	for(batteryNum= 0 ; batteryNum < MAX_BATTERIES ; batteryNum++)
	{
		if (StartBattV[batteryNum]>0.4)
		{
			finished=false;
		}
	}
	// If finished, flash green LED
	while(finished)
	{
		digitalWrite(GreenLED,HIGH);
		delay(2000);
		digitalWrite(GreenLED,LOW);
		delay(2000);
	}
	// run if at least one battery is > 0.4 Volt
	for(batteryNum= 0 ; batteryNum < MAX_BATTERIES ; batteryNum++)
	{
		if ((StartBattV[batteryNum]>0.99) && (digitalRead(TypeSelPin)==HIGH))
		{
			//Stage1, activate both FETs if >0.99V and High current
			digitalWrite(FET1[batteryNum], HIGH);
			delay(200);
			digitalWrite(FET2[batteryNum], HIGH);
		}
		else if (StartBattV[batteryNum]>0.4)
		{
			//Stage 2, activate one FET
			digitalWrite(FET2[batteryNum], HIGH);
			delay(200);
		}
	}	
	while ((millis()-StartTime)<LogInterval); //Do nothing until LogInterval
	// Read LoadBattV and turn off FETs
	for(batteryNum= 0 ; batteryNum < MAX_BATTERIES ; batteryNum++)
	{
		LoadBattV[batteryNum] = getBatteryVoltage(PinV[batteryNum]);
		digitalWrite(FET1[batteryNum], LOW);
		delay(200);
		digitalWrite(FET2[batteryNum], LOW);
	}
	printData();
	delay(2000);
	StartTime=millis(); //Initialize StartTime	
}

//////////////////////////////////////////////////////////////////////////////////
// Read analog input for specified battery and maps into a voltage
//////////////////////////////////////////////////////////////////////////////////
float getBatteryVoltage(unsigned int PinV)
{
	const byte nsamp=10;
	unsigned int VoltValue = 0;  		// value coming from Battery voltage 
	for (j=0;j<nsamp;j++)
		{    
				VoltValue += analogRead(PinV); 
		}    
		return (VoltValue/nsamp)*aRef/1023;
}

//////////////////////////////////////////////////////////////////////////////////
// Calulate Internal Resistance
//////////////////////////////////////////////////////////////////////////////////
float getIntRes(float StartV, float LoadV)
{
	// Internal Resistance is in series with Battery Voltage.
	// Start Voltage is Battery Voltage
	// Load Voltage is voltage across Load Resistor. 
	// Internal Resistor and Load make Voltage divider.
	// I = E/R = LoadV/LoadRes = (StartV - LoadV)/Internal Resistance
	// Internal Res * LoadV = (StartV - LoadV) * LoadRes
	// Internal Res = (StartV*LoadRes)/LoadV - LoadRes
	const float LoadRes=0.8;
	return (StartV*LoadRes/LoadV)-LoadRes;
}

//////////////////////////////////////////////////////////////////////////////////
// Print Data
//////////////////////////////////////////////////////////////////////////////////
void printData()
{
	Serial.print("$, ");
	for(batteryNum= 0 ; batteryNum < MAX_BATTERIES ; batteryNum++)
	{
		Serial.print(StartBattV[batteryNum],2);
		Serial.print(", ");
		Serial.print(LoadBattV[batteryNum],2);
		Serial.print(", ");    
		Serial.print(IntRes[batteryNum],2);
		Serial.print(", ");    
	}	
	Serial.println();
	
	StartTime = millis();	//Get StartTime for next LogInterval
}