/*********************************************************************
This is a program to estimate battery capacitance of 18650 batteries
up to two batteries can be tested at one time
M. Suzuki
Version 2 March 2016
*********************************************************************/
/*********************************************************************
This is an example sketch for our Monochrome Nokia 5110 LCD Displays

  Pick one up today in the adafruit shop!
  ------> http://www.adafruit.com/products/338

These displays use SPI to communicate, 4 or 5 pins are required to
interface

Adafruit invests time and resources providing this open source code,
please support Adafruit and open-source hardware by purchasing
products from Adafruit!

Written by Limor Fried/Ladyada  for Adafruit Industries.
BSD license, check license.txt for more information
All text above, and the splash screen must be included in any redistribution
*********************************************************************/
#include <Adafruit_GFX.h>
#include <Adafruit_PCD8544.h>

float ArefV = 5.1; 		// Supplied 5V
float CutoffV = 2.9;	// Stop discharge 
float Res = 4.0;		// Discharge resistance 
int i;
int dummy;
#define NumBat 2

// LCD5110 connections
// pin 9 - Serial clock out (SCLK)
// pin 10 - Serial data out (DIN)
// pin 11 - Data/Command select (D/C)
// pin 12 - LCD chip select (CS)
// pin 13 - LCD reset (RST)
//  Adafruit_PCD8544(SCLK, DIN, DC, CS, RST);
Adafruit_PCD8544 display = Adafruit_PCD8544(9, 10, 11, 12, 13);

//Digital pins
unsigned int FETCnt[NumBat];	// FET control pins
#define FET1D 6
#define FET2D 5
#define START 8

// Analog pins
unsigned int BatpinA[NumBat];	// Battery voltage
#define Bat1A 1
#define Bat2A 3
unsigned int FETpinA[NumBat];	// FET voltage
#define FET1A 0
#define FET2A 2

float StartVolt[NumBat];
float BatVolt[NumBat];
float FETVolt[NumBat];
unsigned long ChargeAcc[NumBat];	// Current accumulator is in microAmpHours
int BatCapacity[NumBat];			// Battery capacity in mAh (milliAmp hours)
int BatStat[NumBat];

//Enumeration for BatStat
#define NA 0
#define RUN 1
#define PAUSE 2

bool Testing = false;
#define DELTA 10000L		// Length of sample period in milliseconds
unsigned long StartTime;	// When testing starts
unsigned long NextDelta; 	// When delta period is done

//Prototypes for utility functions
float getVolts(unsigned int);	// read analog convert to volts
void addCapacitance();			// add capacitance for another DELTA period
void checkStatus();				// check Batttery status
void displayData();				// display battery data

void setup(){
	// Attach pins to arrays
	BatpinA[0] = Bat1A;
	BatpinA[1] = Bat2A;
	FETpinA[0] = FET1A;
	FETpinA[1] = FET2A;
	FETCnt[0] = FET1D;
	FETCnt[1] = FET2D;
	
	for (i = 0; i < NumBat; i++){	
		dummy=analogRead(BatpinA[i]); 	//discard first analogRead
		dummy=analogRead(FETpinA[i]); 	//discard first analogRead
		pinMode(FETCnt[i], OUTPUT);		// Set for output
	}
	
	pinMode(START, INPUT_PULLUP);		// Set Start button for input
	for (i = 0; i < NumBat; i++){
			digitalWrite(FETCnt[i], LOW);	// Turn off discharge
	}
	
	Serial.begin(115200); 
	
	display.begin();
	display.setContrast(65);	// Choose best contrast
	display.display(); 			// show splashscreen
	delay(20);
	display.clearDisplay();   	// clears the screen and buffer
 }

void loop() {
	if ((digitalRead(START) == HIGH) && !Testing){
		for (i = 0; i < NumBat; i++){
			StartVolt[i] = getVolts(BatpinA[i]);
			// Pullup on battery socket so bat volt must be < 4.5 and > CutoffV
			if ((StartVolt[i] < 4.5)&&(StartVolt[i] > CutoffV))	BatStat[i] = PAUSE;
			else			BatStat[i] = NA;
		}	
		displayData();
		delay(1000);
	}
	else{
		if(!Testing){			//Start button pushed
			StartTime = millis();			// Get StartTime once
			NextDelta = StartTime + DELTA;	// Get NextDelta once
			Testing = true;
			checkStatus();			// Start running
		}
		if (millis() >= NextDelta){	// finished another DELTA, add Capacitance
			checkStatus();			// See if still running
			addCapacitance();
			displayData();
			NextDelta += DELTA;
			delay(2000);
		}
	}
}

//////////////////////////////////////////////////////////////////////////////////
// Read analog input for pinNum averaged over NUM_SAMPLES
// Uses a running average
// Returns a voltage
//////////////////////////////////////////////////////////////////////////////////
float getVolts(unsigned int pinNum){
	const byte NUM_SAMPLES = 3;
	float SumTotal=0;
	for (byte j=0;j<NUM_SAMPLES;j++){    
		SumTotal+= analogRead(pinNum);
		delay(10);
	}    		
	return (SumTotal/NUM_SAMPLES)*ArefV/1023.0;
}
//////////////////////////////////////////////////////////////////////////////////
// addCapacitance() add capacitance for another DELTA period
//////////////////////////////////////////////////////////////////////////////////
void addCapacitance(){
	for (i = 0; i < NumBat; i++){
		if (BatStat[i] == RUN){
			FETVolt[i] = getVolts(FETpinA[i]);
			//(BatVolt[i] - FETVolt[i]) / Res is Amps
			//((BatVolt[i] - FETVolt[i]) / Res) * DELTA) is Amp*mS
			// ChargeAcc is in microAmpHours  
			// Amp*mS *(1000uS/mS) * (1hr/3600S) 
			// so need to multiply * 1000 for milliamps and divide by 3600 
			// for hours so divide by 3.6 to get uAHr
			// added more precision
			ChargeAcc[i] += (((BatVolt[i] - FETVolt[i]) / Res) * DELTA)/0.0036;
			// Calculate capacity, convert uAHr to mAH by dividing by 1000
			BatCapacity[i] = int (ChargeAcc[i] / 1000000L);
		}
	}
}

//////////////////////////////////////////////////////////////////////////////////
// checkStatus() check each Batttery status
//////////////////////////////////////////////////////////////////////////////////
void checkStatus(){
	for (i = 0; i < NumBat; i++){
		if (Testing && (BatStat[i] != NA)){
			BatVolt[i] = getVolts(BatpinA[i]);
			if (BatVolt[i] <= CutoffV){
				digitalWrite(FETCnt[i], LOW);	// Turn off discharge
				BatStat[i] = PAUSE;
			}
			else{
				digitalWrite(FETCnt[i], HIGH);	// Turn on discharge
				BatStat[i] = RUN;
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////////
// displayData()
//////////////////////////////////////////////////////////////////////////////////
void displayData(){
	display.clearDisplay();
	display.setTextSize(1);
	display.setTextColor(BLACK);
	// Line 1 Bat 1 status
	display.print("B1:");
	switch (BatStat[0]) {
		case NA:
			display.println("No Batt");
			break;
		case RUN:
			display.println("Running");
			break;
		case PAUSE:
			display.println("Paused");
			break;
	}
	
	// Line 2 Bat 1 status
	display.print(" ");
	display.print(StartVolt[0],2);
	display.print("V ");
	display.print(BatVolt[0],2);
	display.println("V");
	// Line 3 Bat capacitance
	display.print(" Cap: ");
	display.print(BatCapacity[0]);
	display.println("mAh");
	
	// Line 4 Bat 2 status
	display.print("B2:");
	switch (BatStat[1]) {
		case NA:
			display.println("No Batt");
			break;
		case RUN:
			display.println("Running");
			break;
		case PAUSE:
			display.println("Paused");
			break;
	}
	
	// Line 5 Bat 2 status
	display.print(" ");
	display.print(StartVolt[1],2);
	display.print("V ");
	display.print(BatVolt[1],2);
	display.println("V");
	// Line 6 Bat capacitance
	display.print(" Cap: ");
	display.print(BatCapacity[1]);
	display.println("mAh");
	
	display.display();
}

