/** 
 * SillyClock with display on dual RG (16x8) LED matrix;
 * also has BT (D0, D1), buzzer (A1/D15), relay (A0/D14);
 */

#include <Arduino.h>
#include <Wire.h>
#include "DS1307.h"
#include "pitches.h"


// used to enable/disable Serial.print, time-consuming operation;
// to minimize the code size, all references to Serial should be commented out;
// #define _DEBUG_

//-----------------------------------------------------
// remaining available pins are: A2/D16 and A3/D17
// in some versions of SillyClock the only available pins are A0-A3;
// they are usually used for buttons, tilt sensor, buzzer (which currently is on A1/D15);
// here A0/D14 is used for relay;
#define RELAY_PIN    14   // D14/A0
#define SPEAKER_PIN  15   // D15/A1


//-----------------------------------------------------
// related to time keeping (clock, alarm);
//
// alarm stops beeping after these many minutes;
#define DEFAULT_SNOOZE_MINUTES    2
#define MAX_SNOOZE_MINUTES        5
int snoozeMins = DEFAULT_SNOOZE_MINUTES;

// before the alarm sounds, the relay will be turned on;
#define MINUTES_BEFORE_ALARM   35

// alarm time;
#define DEFAULT_ALARM_HOUR  7
#define DEFAULT_ALARM_MIN   5

// shows if alarm is enabled or not;
boolean isAlarmEnabled = false;
// shows if the alarm is triggered or not;
boolean isAlarmSound = false;
// shows if the relay is on or not;
boolean isRelayOn = false;


// globals; their values get set by calling getTimeFromRTC();
int hour ;//  = 12;
int minute; // = 45;
int second = 0;
int alarmHour = DEFAULT_ALARM_HOUR;
int alarmMin  = DEFAULT_ALARM_MIN;


// receive commands from serial port (BT) in this buffer;
char cmdBuffer[30] = {0};
byte nCrtBufIndex = 0;

// read time from DS1307 at intervals; for 5000, that's about 6 times a second;
#define MAX_TIME_READING_COUNTER  5000
long timeReadingCounter = MAX_TIME_READING_COUNTER;


//--------------------------------------------------------------
// sound related;
//
// TONES  ==========================================
// Start by defining the relationship between 
//       note, period, &  frequency. 
#define  c     3830    // 261 Hz 
#define  d     3400    // 294 Hz 
#define  e     3038    // 329 Hz 
#define  f     2864    // 349 Hz 
#define  g     2550    // 392 Hz 
#define  a     2272    // 440 Hz 
#define  b     2028    // 493 Hz 
#define  C     1912    // 523 Hz 
// Define a special note, 'R', to represent a rest
#define  R     0

// MELODY and TIMING  =======================================
//  melody[] is an array of notes, accompanied by beats[], 
//  which sets each note's relative length (higher #, longer note) 
int melody[] = {  C,  b,  g,  C,  b,   e,  R,  C,  c,  g, a, C };
int beats[]  = { 16, 16, 16,  8,  8,  16, 32, 16, 16, 16, 8, 8 }; 
int MAX_COUNT = sizeof(melody) / 2; // Melody length, for looping.

// Set overall tempo
long tempo = 10000;
// Set length of pause between notes
int pause = 1000;
// Loop variable to increase Rest length
int rest_count = 100; //<-BLETCHEROUS HACK; See NOTES

// Initialize core variables
int note = 0;
int beat = 0;
long duration  = 0;


// a tilt sensor can determine how the digits are displayed;
byte orientation = 1;	// vertical; 1 for horizontal


//--------------------------------------------------------------
// display related;
//
// display colours;
#define BLACK   0
#define RED     1
#define GREEN   2
#define ORANGE  3

// parameters for matrix display;
#define CLOCK_PIN        4
#define LATCH_PIN        5
#define SER_DATA_PIN     6

// pins assigned to LED matrix rows;
byte pinForRow[8] = {8, 9, 10, 11, 12, 13, 7, 3};

// video memory is two arrays of 16 bit integers, one per colour;
volatile uint16_t screenMemR[8] = {0};
volatile uint16_t screenMemG[8] = {0};

// used for blinking a whole 8x8 sprite;
long     blinkCounter = 0;
boolean  blinkLow     = true;
boolean  blinkHigh    = false;
boolean  blinkBlank   = false;

int crtColor = GREEN;

byte soft_prescaler = 0;
byte activeRow      = 0;


// used to display clock digits on half of an 8x8 display;
byte digit[][4] = {
  {0, 0xFE, 0x82, 0xFE},  // 0
  {0, 0x84, 0xFE, 0x80},  // 1
  {0, 0xF2, 0x92, 0x9E},  // 2
  {0, 0x92, 0x92, 0xFE},  // 3
  {0, 0x1E, 0x10, 0xFE},  // 4
  {0, 0x9E, 0x92, 0xF2},  // 5
  {0, 0xFE, 0x92, 0xF2},  // 6
  {0, 0x02, 0xF2, 0x0E},  // 7
  {0, 0xFE, 0x92, 0xFE},  // 8
  {0, 0x9E, 0x92, 0xFE},  // 9
};

// thin digits to fit 2 in 8x8;
byte digitH[10][8] = {
  {0, 2, 5, 5, 5, 5, 2, 0},  // 0
  {0, 2, 6, 2, 2, 2, 7, 0},  // 1
  {0, 6, 1, 2, 4, 4, 7, 0},  // 2
  {0, 6, 1, 2, 1, 1, 6, 0},  // 3
  {0, 4, 5, 7, 1, 1, 1, 0},  // 4
  {0, 7, 4, 6, 1, 1, 6, 0},  // 5
  {0, 3, 4, 6, 5, 5, 2, 0},  // 6
  {0, 7, 1, 2, 2, 2, 2, 0},  // 7
  {0, 7, 5, 2, 5, 5, 2, 0},  // 8
  {0, 2, 5, 5, 3, 1, 6, 0},  // 9
};


byte sprites[][8] = {
  {
    0x00,    // ________   blank
    0x00,    // ________
    0x00,    // ________
    0x00,    // ________
    0x00,    // ________
    0x00,    // ________
    0x00,    // ________
    0x00     // ________
  },
  {
    0x20,    // __X_____  bell
    0x3C,    // __XXXX__
    0x22,    // __X___X_
    0xE1,    // XXX____X
    0xE1,    // XXX____X
    0x22,    // __X___X_
    0x3C,    // __XXXX__
    0x20,    // __X_____
  },
  {
    0x20,    // __X_____  no bell
    0x3C,    // __XXXX__
    0x22,    // __X___X_
    0x21,    // __X____X
    0x21,    // __X____X
    0x22,    // __X___X_
    0x3C,    // __XXXX__
    0x20,    // __X_____
  },
};


void setup()
{
  // Calculation for timer 2
  // 16 MHz / 8 = 2 MHz (prescaler 8)
  // 2 MHz / 256 = 7812 Hz
  // soft_prescaler = 15 ==> 520.8 updates per second
  // 520.8 / 8 rows ==> 65.1 Hz for the complete display
  TCCR2A = 0;           // normal operation
  TCCR2B = (1<<CS21);   // prescaler 8
  TIMSK2 = (1<<TOIE2);  // enable overflow interrupt


  // define outputs for serial shift registers
  pinMode(CLOCK_PIN,    OUTPUT);
  pinMode(LATCH_PIN,    OUTPUT);
  pinMode(SER_DATA_PIN, OUTPUT);

  // set outputs for the 8 matrix rows;
  for (int i=0; i<8; i++)
    pinMode(pinForRow[i], OUTPUT);

  resetDisplay();

  Serial.begin(9600);
  Serial.println("in setup");

  // read parameter values (alarm etc) as saved in RTC's memory;
  initSavedParams();

  pinMode(SPEAKER_PIN, OUTPUT);
  pinMode(RELAY_PIN, OUTPUT);
  
//  playMelody();
  beep2();
  
//  setTime(13, 30, 0);
}



ISR(TIMER2_OVF_vect)
{
  soft_prescaler++;
  if (soft_prescaler == 2)
  {
     //display the next row
    displayActiveRow();
    soft_prescaler = 0;
  }
  //blinkCounter++;
  if (blinkCounter > 5000)
  {
   // blinkCounter = 0;
    blinkBlank = !blinkBlank;
  }
}


void displayActiveRow()
{
  // disable current row;
  digitalWrite(pinForRow[activeRow], LOW);

  // set next row;
  activeRow = (activeRow+1) % 8;

  boolean lowByteIsBlank  = blinkLow  && blinkBlank;
  boolean highByteIsBlank = blinkHigh && blinkBlank;

  // shift out values for this row;
  byte redHi = screenMemR[activeRow] >> 8;
  byte redLo = screenMemR[activeRow] & 0xFF;

  byte greenHi = screenMemG[activeRow] >> 8;
  byte greenLo = screenMemG[activeRow] & 0xFF;

  shiftOutRow(highByteIsBlank ? 0 : redHi, highByteIsBlank ? 0 : greenHi, lowByteIsBlank ? 0 : redLo, lowByteIsBlank ? 0 : greenLo);

  // switch to new row;
  digitalWrite(pinForRow[activeRow], HIGH);
}



void shiftOutRow(byte redHi, byte greenHi, byte redLo, byte greenLo)
{
  digitalWrite(LATCH_PIN, LOW);

  shiftOut(SER_DATA_PIN, CLOCK_PIN, LSBFIRST, ~redHi);
  shiftOut(SER_DATA_PIN, CLOCK_PIN, LSBFIRST, ~greenHi);

  shiftOut(SER_DATA_PIN, CLOCK_PIN, LSBFIRST, ~redLo);   
  shiftOut(SER_DATA_PIN, CLOCK_PIN, LSBFIRST, ~greenLo);   

  // return the latch pin high to signal chip that it 
  // no longer needs to listen for information
  digitalWrite(LATCH_PIN, HIGH);
}


void resetDisplay()
{
  for (byte i = 0; i < 8; i++)  screenMemR[i] = 0;
  for (byte i = 0; i < 8; i++)  screenMemG[i] = 0;
  crtColor = GREEN;
}


void loop()
{
  timeReadingCounter++;
  if (timeReadingCounter > MAX_TIME_READING_COUNTER)
  {
    getTimeFromRTC();
    displayCurrentTime();

    timeReadingCounter = 0;
  }

  checkCommands();  // from BT;

  // check if it's time to trigger the alarm;
  if (isAlarmEnabled)
  {
    if (hour == alarmHour && minute == alarmMin && second == 0)
      isAlarmSound = true;
  }

  if (isAlarmSound)
  {
    // here do something when the alarm was triggered;
    beep2();
    
    // automatically stop the alarm after snooze time;
    int crtTime = hour * 60 + minute;
    int alarmTime = alarmHour * 60 + alarmMin;

    if (crtTime < alarmTime)
    {
      // this is the case when alarm time is before 12:00AM and crt time is just passed that;
      crtTime += 1440;  // 24*60 minutes
    }
    
    // turn off both the buzzer and the relay;
    if (crtTime >= alarmTime + snoozeMins)
    {
        isAlarmSound = false;
        isRelayOn    = false;
    }
  }
  
  // a few minutes before alarm is triggered turn on the relay (to brew the coffee, for example);
  isRelayOn = checkTimeForRelay();
  digitalWrite(RELAY_PIN, (isRelayOn? HIGH : LOW));
}


// statically displays the given character/sprite;
void setScreenMem(byte color, byte sprite[8])
{
  uint16_t row;
  for (byte i = 0; i < 8; i++)
  {
    row = sprite[i];
    if ((color & RED)   == RED)    screenMemR[i] = row;
    if ((color & GREEN) == GREEN)  screenMemG[i] = row;
  }
}


void setScreenMem(byte color, int hour, int minute)
{
  int h1 = hour / 10;
  int h2 = hour % 10;
  int m1 = minute / 10;
  int m2 = minute % 10;

  for (byte i = 0; i < 8; i++)
  {
    uint16_t row = 0;
    if (i>3)
    {
      row = (((uint16_t) (digit[m2][i-4])) << 8) + digit[h2][i-4];
    }
    else
    {
      row = (((uint16_t) (digit[m1][i])) << 8) + (h1 ? digit[h1][i] : 0);
    }
    
    if ((color & RED)   == RED )  screenMemR[i] = row;
    if ((color & GREEN) == GREEN) screenMemG[i] = row;
  }
}


void setScreenMemHorizontal(byte color, int hour, int minute)
{
  int h1 = hour / 10;
  int h2 = hour % 10;
  int m1 = minute / 10;
  int m2 = minute % 10;

  for (byte i = 0; i < 8; i++)
  {
    uint16_t row = 0;
    if (h1 == 0)
      row = (uint16_t) ((digitH[h2][i] << 9) + (digitH[m1][i] << 4) + digitH[m2][i]);
    else
      row = (uint16_t) ((digitH[h1][i] << 13) + (digitH[h2][i] << 9) + (digitH[m1][i] << 4) + digitH[m2][i]);
    
    if ((color & RED) == RED)      screenMemR[i] = row;
    if ((color & GREEN) == GREEN)  screenMemG[i] = row;
  }
}



//------------------------------------------------------------
// read data stored in RTC's RAM;
void initSavedParams()
{
    readAlarmTime();
    snoozeMins       = getSnoozeTime();
    isAlarmEnabled   = getAlarmOnOff();

// when powered for the first time, the values are not there (nonsense is returned);
// we need to initialize them with defaults and save them;
  if (alarmHour > 23 || alarmMin > 59)
  {
    // at least one param came as out-of-bounds, probably because it was never written to memory;
    snoozeMins = DEFAULT_SNOOZE_MINUTES;
    saveSnoozeTime();

    alarmHour = DEFAULT_ALARM_HOUR;
    alarmMin  = DEFAULT_ALARM_MIN;
    saveAlarmTime();
    
    isAlarmEnabled = false;
    saveAlarmOnOff();
  }

#ifdef _DEBUG_
  Serial.println("Params restored from RTC RAM: ");
  Serial.print("Alarm time: ");
  Serial.print(alarmHour);
  Serial.print(":");
  Serial.println(alarmMin);
  Serial.print("Is alarm enabled: ");
  Serial.println(isAlarmEnabled, HEX);
#endif
}


//------------------------------------------------------------
// Read from (write to) the DS1307 RAM;

// store settings in RTC's memory at these locations;
// locations are compatible with those used in Wise Clock 2;
#define alarmHoursLoc  0     // alarm hours storage location
#define alarmMinsLoc   1     // alarm minutes storage location
#define alarmOnOffLoc  2     // alarmOn storage location
#define alarmModeLoc   3     // alarmMode storage location
#define clockAMPMLoc   5     // clock AMPM vs 24hr storage location
#define speedLoc       6     // speed storage location
#define snoozeTimeLoc  7     // snooze time storage location
#define showDateLoc    8     // "show date" storage location
#define showMessageLoc 9     // "show message" storage location


void EEwrite_RTC(int loc, int value)
{
    RTC_DS1307.set_sram_byte(byte(value), loc);
    delay(10);
}

int EEread_RTC(int loc)
{
    return int(RTC_DS1307.get_sram_byte(loc));
}


// Read and write alarm status
boolean getAlarmOnOff()
{
    return boolean( EEread_RTC(alarmOnOffLoc) );
}

void saveAlarmOnOff()
{
    EEwrite_RTC( alarmOnOffLoc, isAlarmEnabled); 
}


// Read and write alarm time (hour, minute);
void readAlarmTime()
{
    alarmMin  = EEread_RTC( alarmMinsLoc );
    alarmHour = EEread_RTC( alarmHoursLoc );
}

void saveAlarmTime()
{
    EEwrite_RTC( alarmMinsLoc,  alarmMin );
    EEwrite_RTC( alarmHoursLoc, alarmHour );
#ifdef _DEBUG_
  Serial.print("Alarm time saved to RTC RAM: ");
  Serial.print(alarmHour);
  Serial.print(":");
  Serial.println(alarmMin);
#endif
}


// Read and write snooze time
int getSnoozeTime()
{
    return EEread_RTC( snoozeTimeLoc ); 
}

void saveSnoozeTime()
{
    EEwrite_RTC( snoozeTimeLoc, snoozeMins ); 
#ifdef _DEBUG_
  Serial.print("Saved snooze time: ");
  Serial.println(snoozeMins);
#endif
}


void getTimeFromRTC()
{
  int rtc[7];
  RTC_DS1307.get(rtc, true);

  second = rtc[0];
  minute = rtc[1];
  hour   = rtc[2];

#ifdef _DEBUG_
    Serial.print("Time is ");
    Serial.print(hour);
    Serial.print(":");
    Serial.print(minute);
    Serial.print(":");
    Serial.println(second);
#endif
}


void setTime(int hh, int mm, int ss)
{
  RTC_DS1307.stop();
  RTC_DS1307.set(DS1307_SEC,  ss);
  RTC_DS1307.set(DS1307_MIN,  mm);
  RTC_DS1307.set(DS1307_HR,   hh);
/*
  RTC_DS1307.set(DS1307_DOW,  1);
  RTC_DS1307.set(DS1307_DATE, day);
  RTC_DS1307.set(DS1307_MTH,  month);
  RTC_DS1307.set(DS1307_YR,   year);
*/
  RTC_DS1307.start();
}


void displayCurrentTime()
{
      byte color = crtColor;

      // since time may have changed, find out what colour will be displayed with;
      // depending on how close the time is to the alarm time, color could be:
      // red    - one hour and less;
      // yellow - three hours to one hour
      // green  - more than three hours;
      // the colors are applied only when alarm is enabled;
      if (isAlarmEnabled)
      {
          int time      = hour * 60  + minute;
          int alarmTime = alarmHour * 60 + alarmMin;

          // once passed the alarm time, look at the next one;
          if (time > alarmTime) alarmTime += 1440;  // 24*60
          if (alarmTime - time > 180)  // more than 3 hours to alarm;
          {
            color = GREEN;
          }
          else if (alarmTime - time > 60)  // more than 1 hour to alarm;
          {
            color = ORANGE;
          }
          else  // less than 1 hour to alarm;
          {
            color = RED;
          }
      }
      else  // always green if alarm not enabled;
      {
        color = GREEN;
      }
      
      // blink display while alarm sounds;
      blinkLow = blinkHigh = isAlarmSound;

      if (color != crtColor)
      {
        resetDisplay();
        crtColor = color;
      }

      // used with a tilt sensor;
      if (orientation == 0)
        setScreenMem(crtColor, hour, minute);
      else
        setScreenMemHorizontal(crtColor, hour, minute);
}


void playNote()
{
  long elapsed_time = 0;
  if (note > 0)
  {
    // if this isn't a Rest beat, while the tone has 
    //  played less long than 'duration', pulse speaker HIGH and LOW
    while (elapsed_time < duration)
    {
      digitalWrite(SPEAKER_PIN, HIGH);
      delayMicroseconds(note / 2);

      // DOWN
      digitalWrite(SPEAKER_PIN, LOW);
      delayMicroseconds(note / 2);

      // Keep track of how long we pulsed
      elapsed_time += (note);
    } 
  }
  else
  {
    // Rest beat; loop times delay
    for (int j = 0; j < rest_count; j++)
      delayMicroseconds(duration);  
  }                                 
}


void playMelody()
{
  // Set up a counter to pull from melody[] and beats[]
  for (int i=0; i<MAX_COUNT; i++)
  {
    note = melody[i];
    beat = beats[i];
    duration = beat * tempo; // Set up timing
    playNote(); 
    // pause between notes...
    delayMicroseconds(pause);
  }
}


void checkCommands()
{
	while (Serial.available() > 0)
	{
		// read the incoming byte;
		char inChar = Serial.read();
		cmdBuffer[nCrtBufIndex++] = inChar;
		if (nCrtBufIndex >= sizeof(cmdBuffer)-1)
                    shiftBufferLeft();
	}

        if (0 == strncmp(cmdBuffer, "ALARM TIME=", 11) && nCrtBufIndex > 15)
        {
          beep2();
          #ifdef DEBUG
            Serial.print(">>Received command: ");
            Serial.println(cmdBuffer);
          #endif

          // next characters are the alarm time, formatted HH:MM;
          alarmHour = (cmdBuffer[11]-'0') * 10 + (cmdBuffer[12]-'0');
          alarmMin  = (cmdBuffer[14]-'0') * 10 + (cmdBuffer[15]-'0');

          saveAlarmTime();
          resetBuffer();
        }
	else if (0 == strncmp(cmdBuffer, "TIME=", 5) && nCrtBufIndex > 9)
	{
          // time setting command;
          beep2();
          #ifdef DEBUG
            Serial.print(">>Received command: ");
            Serial.println(cmdBuffer);
          #endif

          // next characters are the time, formatted HH:MM;
          hour   = (cmdBuffer[5]-'0') * 10 + (cmdBuffer[6]-'0');
          minute = (cmdBuffer[8]-'0') * 10 + (cmdBuffer[9]-'0');

          setTime(hour, minute, 1);
          resetBuffer();
	}
        else if (0 == strncmp(cmdBuffer, "STOP ALARM", 10))
        {
          isAlarmSound = false;
          isRelayOn    = false;
          resetBuffer();
        }
        else if (0 == strncmp(cmdBuffer, "ALARM OFF", 9))
        {
          beep2();
          isAlarmEnabled = false;
          saveAlarmOnOff();
          resetBuffer();
        }
        else if (0 == strncmp(cmdBuffer, "ALARM ON", 8))
        {
          beep2();
          isAlarmEnabled = true;
          saveAlarmOnOff();
          resetBuffer();
        }
        else if (0 == strncmp(cmdBuffer, "SHOW ALARM", 10))
        {
          printAlarmTime();
          resetBuffer();
        }
}


void resetBuffer()
{
  for (byte i=0; i<sizeof(cmdBuffer); i++)
    cmdBuffer[i] = 0;

  nCrtBufIndex = 0;
}

void shiftBufferLeft()
{
  for (byte i=0; i<sizeof(cmdBuffer)-1; i++)
    cmdBuffer[i] = cmdBuffer[i+1];

  nCrtBufIndex--;
}


void chime()
{
  // used at the top and bottom of the hour;
  tone(SPEAKER_PIN, NOTE_B5, 25);
}

void beep2()
{
    tone(SPEAKER_PIN, NOTE_C4, 100);
}

void printTime()
{
	Serial.print("Time: ");
	Serial.print(hour);
	Serial.print(":");
	Serial.print(minute);
	Serial.print(":");
	Serial.print(second);
	Serial.println("");
}

void printAlarmTime()
{
	Serial.print("Alarm is ");
	Serial.print(isAlarmEnabled? "ON" : "OFF");
	Serial.print("; Alarm Time: ");
	Serial.print(alarmHour);
	Serial.print(":");
	Serial.print(alarmMin);
	Serial.println("");
}


boolean checkTimeForRelay()
{
  boolean isLessThan15Minutes = false;

  if (isAlarmEnabled)
  {
    int crtTime = hour * 60 + minute;
    int alarmTime = alarmHour * 60 + alarmMin;
    if (alarmTime < crtTime)
    {
      // the case when alarm time is just after 12:00AM (00:mm) and time is just before 12:00AM (23:mm);
      alarmTime += 1440;  // 24*60
    }
    
    if (alarmTime - crtTime < MINUTES_BEFORE_ALARM)
    {
      isLessThan15Minutes = true;
    }
  }

  return isLessThan15Minutes;
}
