/* 
An auto Leveler for the Arduino
Written by Michael Suzuki
*/

#include <Stepper.h>				//Stepper library
#include <avr/EEPROM.h>				// Store NS, EW Center values

// These are addresses into EEPROM memory.  
//The values to be stores are floats which 
// need 4 bytes each.  Thus 0,4,8,12,...
#define NS_CENT_ADR 	0
#define EW_CENT_ADR 	4

#define MAX_AX        	2			// NS ,EW
#define NS				0
#define EW				1

const float aRef=3.01;				//Aref
const float hysterisis=0.007;

//Accelerometer Center values
const float NSCenter=1.577;			
const float EWCenter=1.733;
const float AllowedOffset=0.2;
unsigned int AxNum;
unsigned int CENT_ADDR[MAX_AX];	//Center EEPROM addresses
float Center[MAX_AX];				//Center values
float AxisV[MAX_AX];				// Accelerometer Voltages
float CenterV[MAX_AX];			// Calibrated 'balanced' voltages
float Aver[MAX_AX];
	
//Stepper motor constants
const int stepsPerRevolution = 48;  // Step is 7.5 degrees
const int stepRate=12;	
const int stepDelay=500;
// initialize the steppers
Stepper EWStepper(stepsPerRevolution, 8,7,6,5); 
Stepper NSStepper(stepsPerRevolution, 12,11,10,9); 
          
byte j=0;							// for loop counter
byte k=0;							// for loop counter
unsigned long startTime;
unsigned long runTime=15*1000;		// runTime in milliseconds
boolean calSet=false;				// if calibration is set

// Analog pins
unsigned int PinAccel[MAX_AX];	// X(NS) and Y(EW) voltage pins from Accelerometer

// Digital pins
// D5,6,7,8,9,10,11,12 used by steppers
unsigned int CalSelPin;				// Switch if low, then calibrate mode
unsigned int AutoCalPin;			// Switch if low, then AutoCalibrate mode
unsigned int LaserPowerPin;			// High turns on Laser Power

//Prototypes for utility functions
float getAxisV(unsigned int pinN);
float readFloat(int address);
void writeFloat(float value,int address);

void setup() {
	analogReference(EXTERNAL); //3.3V connected to AREF thru 3.3K AREF=3VDC
	//discard first analogReads
	for (j=0; j <= 5; j++){
		int dummy=analogRead(j);
	}
	
	//Digital setup
	CalSelPin=4;					// If low, then calibration mode
	AutoCalPin=3;					// If low, then autocalibrate
	pinMode(CalSelPin, INPUT);      // set CalSel to input
	pinMode(AutoCalPin, INPUT);     // set AutoCal to input
	LaserPowerPin=13;				// If HIGH then power on
	pinMode(LaserPowerPin, OUTPUT);
	
	digitalWrite(CalSelPin, HIGH);  // turn on 20K pullup resistor
	digitalWrite(AutoCalPin, HIGH); // turn on 20K pullup resistor
	// Assign analog pin numbers
	PinAccel[NS]=4;					//X North-South
	PinAccel[EW]=5;					//Y East-West

	CENT_ADDR[NS]=NS_CENT_ADR;
	CENT_ADDR[EW]=EW_CENT_ADR;
	Center[NS]=NSCenter;
	Center[EW]=EWCenter;
	
	// set the speeds TO 6 rpm:
	EWStepper.setSpeed(6);
	NSStepper.setSpeed(6);
	
	// initialize the serial port:
	Serial.begin(9600);
	
	// Write defaults if not within range, e.g., new Arduino
	for(AxNum= 0 ; AxNum < MAX_AX ; AxNum++){	
		if ((CenterV[AxNum]<NSCenter-AllowedOffset)||(CenterV[AxNum]>NSCenter+AllowedOffset)){
			writeFloat(Center[AxNum],CENT_ADDR[AxNum]); //This should only happen the first time
			CenterV[AxNum]=readFloat(CENT_ADDR[AxNum]);
		}
	}
	Serial.print("NS Center is ");
	Serial.print(CenterV[NS],3);
	Serial.print(", EW Center is ");
	Serial.println(CenterV[EW],3);
}

void loop(){
	if(digitalRead(CalSelPin)==LOW){		//Calibration mode, display average
		// Get voltages
		for(AxNum= 0 ; AxNum < MAX_AX ; AxNum++){
			AxisV[AxNum] = getAxisV(PinAccel[AxNum]);
		}
		Serial.print("NS Volt is ");
		Serial.print(AxisV[NS],3);
		Serial.print(", EW Volt is ");
		Serial.println(AxisV[EW],3);
		delay(2000);
	}
	else{										
		if(digitalRead(AutoCalPin)==LOW){		// AutoCal mode
			const byte nsamp=25;
			for (k=0;k<nsamp;k++){ 
				for(AxNum= 0 ; AxNum < MAX_AX ; AxNum++){			
					Aver[AxNum] += getAxisV(PinAccel[AxNum]);
				}
			}    
			for(AxNum= 0 ; AxNum < MAX_AX ; AxNum++){
				Aver[AxNum] = Aver[AxNum]/nsamp;	// Get Average
				if ((Aver[AxNum]>NSCenter-AllowedOffset)&&(Aver[AxNum]<NSCenter+AllowedOffset)){
					writeFloat(Aver[AxNum],CENT_ADDR[AxNum]); //Write new value to EEPROM
					CenterV[AxNum]=readFloat(CENT_ADDR[AxNum]);
				}
			}
			Serial.print("NS Center is ");
			Serial.print(CenterV[NS],3);
			Serial.print(", EW Center is ");
			Serial.println(CenterV[EW],3);
			while(1);							//Stop
		}
		else{									// Running mode
			// Get voltages
			for(AxNum= 0 ; AxNum < MAX_AX ; AxNum++){
				AxisV[AxNum] = getAxisV(PinAccel[AxNum]);
			}
			Serial.print("NS Volt is ,");
			Serial.print(AxisV[NS],3);
			Serial.print(", EW Volt is , ");
			Serial.println(AxisV[EW],3);
			delay(1000);

			calSet=true;
			//NS
			while (AxisV[NS] >(CenterV[NS]+hysterisis)){
				calSet=false;
				NSStepper.step(stepRate);		//Positive CW is up
				delay(stepDelay);
				//dummyRead = getAxisV(PinAccel[NS]);	//Discard first sampling
				AxisV[NS] = getAxisV(PinAccel[NS]);
				Serial.print("NS Volt is ");
				Serial.print(AxisV[NS],3);
				Serial.print(" > CenterV+Hyst is ");
				Serial.println((CenterV[NS]+hysterisis),3);
			}
			while (AxisV[NS] <(CenterV[NS]-hysterisis)){
				calSet=false;
				NSStepper.step(-stepRate);		//Negative CCW is down
				delay(stepDelay);
				delay(stepDelay);
				AxisV[NS] = getAxisV(PinAccel[NS]);
				Serial.print("NS Volt is ");
				Serial.print(AxisV[NS],3);
				Serial.print(" < CenterV-Hyst is ");
				Serial.println((CenterV[NS]-hysterisis),3);
			}

			//EW
			while (AxisV[EW] >(CenterV[EW]+hysterisis)){
				calSet=false;
				EWStepper.step(-stepRate);		//Negative CCW is down
				delay(stepDelay);
				//dummyRead = getAxisV(PinAccel[EW]);	//Discard first sampling
				delay(stepDelay);
				AxisV[EW] = getAxisV(PinAccel[EW]);
				Serial.print("EW Volt is ");
				Serial.print(AxisV[EW],3);
				Serial.print(" > CenterV+Hyst is ");
				Serial.println((CenterV[EW]+hysterisis),3);
			}
			while (AxisV[EW] <(CenterV[EW]-hysterisis)){
				calSet=false;
				EWStepper.step(stepRate);		//Positive CW is up
				delay(stepDelay);
				delay(stepDelay);
				AxisV[EW] = getAxisV(PinAccel[EW]);
				Serial.print("EW Volt is ,");
				Serial.println(AxisV[EW],3);
				Serial.print(" < CenterV-Hyst is .");
				Serial.println((CenterV[EW]-hysterisis),3);
			}
			
			if (calSet){									// Both axis are okay
				startTime=millis();
				digitalWrite(LaserPowerPin, HIGH);		//Turn on Laser		
				while((millis()-startTime)<runTime);	//do nothing until runTime finished
				digitalWrite(LaserPowerPin, LOW);		//Turn off Laser
				delay(2000);							//allow power to settle
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////////
// Read analog input for specified Axis and maps into a voltage
//////////////////////////////////////////////////////////////////////////////////
float getAxisV(unsigned int pinN){
	const byte nsamp=10;
	unsigned int VoltValue = 0;  		// value coming from Axis voltage 
	// Get nsamps (10)
	for (j=0;j<nsamp;j++){    
				VoltValue += analogRead(pinN); 
				delay(5);				// This improved repeatability
		}    
		return (VoltValue/nsamp)*aRef/1023;
}

//////////////////////////////////////////////////////////////////////////////////
// Read EEPROM Float value
//////////////////////////////////////////////////////////////////////////////////
float readFloat(int address){
  float out;
  eeprom_read_block((void *) &out, (unsigned char *) address ,4 );
  return out;
}
//////////////////////////////////////////////////////////////////////////////////
// Write EEPROM Float value
//////////////////////////////////////////////////////////////////////////////////
void writeFloat(float value, int address){
  eeprom_write_block((void *) &value, (unsigned char *) address ,4);
}

