//  Copyright (C) 2012 James Coliz, Jr. <maniacbug@ymail.com>

/**
 * L.O.G. get GPS coordinates, 
 * send to base station via 
 * nrf24L01
 */

#include <TinyGPS++.h>
#include <SoftwareSerial.h>
#include <RF24Network.h>
#include <RF24.h>
#include <SPI.h>

// GPS connections (RX pin connects to TX on module and v.v.) and Baud rate
static const int RXPin = 2, TXPin = 3;
static const uint32_t GPSBaud = 9600;
// The serial connection to the GPS device
SoftwareSerial sGPS(RXPin, TXPin);

// The TinyGPS++ object
TinyGPSPlus gps;

// GPS Latitude[0]/Longitude[1]
double myLL[2];		//current GPS location


// nRF24L01(+) radio attached  (CE, CSN)
RF24 radio(14,10);
// Network uses that radio
RF24Network network(radio);
// Channel of our node
const uint16_t channel = 60;
// Address of our node
const uint16_t this_node = 1;
// Address of the base
const uint16_t other_node = 0;

// How many packets have we sent already
unsigned long packets_sent;

// Structure of our payload, limited to 32 bytes
struct payload_t			// 32 bytes max
{
  unsigned long counter;	// 4 bytes
  double lat;				// 4 bytes
  double lng;				// 4 bytes
  float Vcc;				// 4 bytes
};

// Analog pins
#define VccPin 2
// Digital pins
#define ActivePin 9

float Vcc; 	//Supplied Voltage

// Prototypes
void getGPS();							// get GPS data
static void smartDelay(unsigned long);	// ensures that the gps object is being "fed".
void getVoltage();						// getVoltage
void sendPayload();						// check if time to send payload

void setup(void)
{
	analogReference(INTERNAL); 		// Set analog reference to 1.1V
	analogRead(VccPin); 			//discard first analogRead
	pinMode(ActivePin, OUTPUT);		// Set for output
 
 	// start Software Serial
	sGPS.begin(GPSBaud);

Serial.begin(115200); 
	SPI.begin();
	radio.begin();
	network.begin(channel, this_node);

	// Power down the radio. Note that the radio will get powered back up on the next write() call.
	radio.powerDown();
}

void loop(void){
	// Pump the network regularly
	network.update();
	while (sGPS.available() > 0){
		if (gps.encode(sGPS.read())){
			getGPS();
			getVoltage();
			sendPayload();
			smartDelay(2000);
		}
	}
}

//////////////////////////////////////////////////////////////////
// getGPS
//////////////////////////////////////////////////////////////////
void getGPS(){
	if (gps.location.isValid()){
		myLL[0] = gps.location.lat();
		myLL[1] = gps.location.lng();
/*Serial.print("lat: ");
Serial.println(myLL[0],7);
Serial.print("lng: ");
Serial.println(myLL[1],7);
*/
	}
}

//////////////////////////////////////////////////////////////////////////////////
// Read analog input for VccPin averaged over NUM_SAMPLES
// Uses a running average
// Vcc is scaled with a voltage divider * 75K/(75K + 240K) so reverse
// Should be 4.2, try 3.9
//////////////////////////////////////////////////////////////////////////////////
void getVoltage(){
	const byte NUM_SAMPLES = 20;
	float SumTotal=0;
	for (byte j=0;j<NUM_SAMPLES;j++){    
		SumTotal+= analogRead(VccPin);
		delay(10);
	}    		
	Vcc =  ((SumTotal/NUM_SAMPLES)*1.1/1023.0)*4.13;
	Serial.println(Vcc);
}
//////////////////////////////////////////////////////////////////////////////////
// sendPayload();					// send payload
//////////////////////////////////////////////////////////////////////////////////
void sendPayload(){
	digitalWrite(ActivePin, HIGH);		// Turn on LED
	payload_t payload = { packets_sent++, myLL[0], myLL[1], Vcc };
	RF24NetworkHeader header(/*to node*/ other_node);
	bool ok = network.write(header,&payload,sizeof(payload));
	// Power down the radio. Note that the radio will get powered back up on the next write() call.
	radio.powerDown();
	digitalWrite(ActivePin, LOW);		// Turn off LED
}
// This custom version of delay() ensures that the gps object
// is being "fed".
static void smartDelay(unsigned long ms){
  unsigned long start = millis();
  do 
  {
    while (sGPS.available())
      gps.encode(sGPS.read());
  } while (millis() - start < ms);
}

