//NOTE: The ATtiny85 must be set to use the 8-MHz internal clock
//      This requires "Burn Bootloader" to be run first

// Receives signals from 433-MHz motion detectors
// Must know the specific codes being transmitted
// Message length: 24 data bits, 1 stop bit
// Written by Boomer48: September 2019

#define Pulse_In 1
#define RX_LED 0 // Latched on if any motion detected
#define Play_E 4 // Triggers sound module every time motion is detected
#define Msg_Length 3 // Defined message length for specific transmitter

// All times in microseconds
#define Min_Gap_Time 9000 // nominal = 9600
#define Max_Gap_Time 10200 // nominal = 9600
#define One_Bit_Time 260 // nominal = 320
#define Zero_Bit_Time 900 // nominal = 960

// Defines for expected signals
#define MD1_1 0xF1
#define MD1_2 0x29
#define MD1_3 0xA5 // Motion Detector 1
#define MD2_1 0xB6
#define MD2_2 0x20
#define MD2_3 0x15 // Motion Detector 2

byte RF_Bit_Count;
byte RF_Byte_Count;
byte RF_Byte = 0;
byte RF_Message[] = {0, 0, 0}; // 24 bits
unsigned long Start_Time = 0;
int Pulse_Width;
byte Started = false;

void setup() {
  pinMode(RX_LED, OUTPUT);
  digitalWrite(RX_LED, 0);
  pinMode(Play_E, OUTPUT);
  digitalWrite(Play_E, 0);
  digitalWrite(Pulse_In, HIGH); //turn on pullup

  // Set interrupt-on-change registers
  bitClear(GIMSK, INT0); // disable INT0 external interrupt
  bitSet(GIMSK, PCIE); // enable interrupt-on-change
  bitSet(PCMSK, Pulse_In);
}

void loop() {
  byte i;

  if (RF_Byte_Count == Msg_Length) {
    noInterrupts();

    if ((RF_Message[0] == MD1_1) && (RF_Message[1] == MD1_2) && (RF_Message[2] == MD1_3))
    {
      digitalWrite(Play_E, 1);
      digitalWrite(RX_LED, 1);
      digitalWrite(Play_E, 0);
    }

    if ((RF_Message[0] == MD2_1) && (RF_Message[1] == MD2_2) && (RF_Message[2] == MD2_3))
    {
      digitalWrite(Play_E, 1);
      digitalWrite(RX_LED, 1);
      digitalWrite(Play_E, 0);
    }

    Started = false;
    interrupts();
  }
} // end loop

// Interrupt-on-change handler
ISR (PCINT0_vect)
{
  //when the pin goes LOW record the pulse start time
  if (digitalRead(Pulse_In) == LOW) {
    Start_Time = micros();
  }
  else // pin went high
    if (Start_Time != 0) {
      //calculate the pulse width
      Pulse_Width = ((volatile int)micros() - Start_Time);
      //clear the timer
      Start_Time = 0;

      // Now check for valid message
      if (Pulse_Width < Max_Gap_Time) {
        if (Started == true) {
          if (Pulse_Width > Zero_Bit_Time) {
            // bit = 0
            RF_Bit_Count--; // bits received MSB first
            bitClear(RF_Message[RF_Byte_Count], RF_Bit_Count);
            if (RF_Bit_Count == 0) {
              RF_Byte_Count++;
              RF_Bit_Count = 8;
            }
          }
          else if (Pulse_Width > One_Bit_Time) {
            // bit = 1
            RF_Bit_Count--; // bits received MSB first
            bitSet(RF_Message[RF_Byte_Count], RF_Bit_Count);
            if (RF_Bit_Count == 0) {
              RF_Byte_Count++;
              RF_Bit_Count = 8;
            }
          }
          else { // bad bit
            Started = false;
          }
        } // Started

        else if (Pulse_Width > Min_Gap_Time) {
          // start of message
          Started = true;
          RF_Byte_Count = 0;
          RF_Bit_Count = 8;
        }
      }
      else { // invalid message
        Started = false;
      }
    }
}
