/* Wireless mesh for Arduino and APC220 modules
Each node samples two analog voltages. Each node can output two analog voltages.
Nodes all work on the same RF frequency. They are synchronised so each transmits in a particular time slot.
Time signals are generated by node zero, and if node zero does not exist, by the lowest numbered node.
There are 16 nodes in the mesh. Each timing cycle is 4096ms, and so it takes around 4 to 80 seconds to send data through the mesh.
Each node also transmits its local time (sync within a few ms) so nodes that cannot directly talk to node 0 can still synchronise to the mesh.
Each node samples its analog inputs, and puts a time stamp on these values.
Each node stores all the values of all other nodes, so 16 nodes, 2 values per node is 32 values.
When a node is ready to transmit, it sends all 32 values, with associated time stamps.
When receiving, if a value has a newer time stamp, then replace it.
This means that newer values propogate through the mesh. 
Nodes can be paired together, so adjusting a potentiometer on one node will change the voltage on another node (with 4 to 80s delay)
Each node can either be a repeater, or input values, or output values, or all of these at once. 
The mesh can tolerate repeater nodes being removed. Ideally there are several paths data can take.
APC220 modules have open air ranges of up to 1000m. Trees and metal (eg a shed) can greatly decrease this though, to as low as 50m.
The mesh allows signals to get around buildings and barriers.
The 4096ms time slot was calculated as the time needed to send all node data at 9600 baud, using human readable hex (ascii 32 to 126)
and allowing for delays of about 150ms between each packet. This means there are two bytes per value (eg F1) and theoretically
it would be possible to send this as a single byte. However, then ascii 13 can't be used as an end of line marker, and then packets
have to be constructed with start and finish bytes and they get more complex to code. 
Current consumption of a node is about 25mA for the Arduino, 30mA for the radio module and 40mA for the display.
Arduino nodes powered using solar power are run with a swithching regulator stepping 13V down to 7V which means current draw from
a 12V battery is about half the current going into the arduino. Nodes draw roughly 1 watt.
Solar powered nodes are run from a 20W to 80W solar panel, which is oversized, but allows for energy collection even on cloudy days.
The solar panel goes into a $10 charge regulator which has over and undervoltage protection and temperature adjusts the float voltage.
Hopefully a 12V 7Ah SLA alarm battery will have a reasonable life as it is only being discharged a small percentage.
*/ 

/* Rewritten for the ILI9314 display - these have more lines than a 20x4 LCD display and can do graphics as well if needed 
Search "bodmer" on Instructables 
Combine Terminal demo and Ring Meter demo into one code - can display text or graphics
*/

/* To save space, in the Adafruit_gfx_as folder, there is a file load_fonts.txt. Uncomment fonts that are not needed to save memory. Font 2 and 4 are small and medium
if running out of memory can remove font 4*/

#include <Adafruit_GFX_AS.h>     // Core graphics library
#include <Adafruit_ILI9341_AS.h> // Hardware-specific library
#include <SPI.h>

#include <SoftwareSerial.h>

SoftwareSerial radioSerial(2,3); // RX, TX

// ********************** Display variables ****************************************

// These are the ILI9341 pins used for the UNO, we must use hardware SPI
#define _sclk 13
#define _miso 12 // Not used
#define _mosi 11
#define _cs 10
#define _rst 7
#define _dc  9

// Must use hardware SPI for speed
Adafruit_ILI9341_AS tft = Adafruit_ILI9341_AS(_cs, _dc, _rst);

// The scrolling area must be a integral multiple of TEXT_HEIGHT
#define TEXT_HEIGHT 16 // Height of text to be printed and scrolled
#define BOT_FIXED_AREA 0 // Number of lines in bottom fixed area (lines counted from bottom of screen)
#define TOP_FIXED_AREA 16 // Number of lines in top fixed area (lines counted from top of screen)

// The initial y coordinate of the top of the scrolling area
uint16_t yStart = TOP_FIXED_AREA;
// yArea must be a integral multiple of TEXT_HEIGHT
uint16_t yArea = 320-TOP_FIXED_AREA-BOT_FIXED_AREA;
// The initial y coordinate of the top of the bottom text line
uint16_t yDraw = 320 - BOT_FIXED_AREA - TEXT_HEIGHT;

// Keep track of the drawing x coordinate
uint16_t xPos = 0;
// For the byte we read from the serial port
byte data = 0;

// A few test varaibles used during debugging
boolean change_colour = 1;
boolean selected = 1;

// We have to blank the top line each time the display is scrolled, but this takes up to 13 milliseconds
// for a full width line, meanwhile the serial buffer may be filling... and overflowing
// We can speed up scrolling of short text lines by just blanking the character we drew
int blank[19]; // We keep all the strings pixel lengths to optimise the speed of the top line blanking

// ********************************** Code variables ****************************


byte myNode = 2; // my node number
byte partnerNode = 1; // output the value of this node (can output more than one if needed)


// global variables, keeps the stack small and more predictable if all variables are global
char screen[80]; // could fill with ={32,32,32} etc but takes memory and gets overwritten quickly as characters come in
int col = 0; // 0 to 19 column number
unsigned long timeCentral = 0xFFFFF800; // synchronise with node 0, as can't reset the millis counter. start high so can spot rollover errors (otherwise every 49 days)
unsigned long timeOffset  = 0xFFFFF800; // offset from my local millis counter - numbers set so if program node 0, it does a refresh immediately

String radioString = ""; 
String txString = "";
int nodeValuesA0[16]; // 16 nodes in a mesh
int nodeValuesA1[16]; // A1 values
unsigned long nodeTimestamps[16]; // timestamps for each value
int lowestNode = 255; // 0 to 15, 255 is false, 0-15 is node last got a time sync signal from

int inPacketLen = 0;

boolean displayMeter = true; // false displays text from the mesh, true displays just the meter, updating every 2 minutes or so

// *********************** Graphics Meter variables ************

// Meter colour schemes
#define RED2RED 0
#define GREEN2GREEN 1
#define BLUE2BLUE 2
#define BLUE2RED 3
#define GREEN2RED 4
#define RED2GREEN 5

#define ILI9341_GREY 0x2104 // Dark grey 16 bit colour


// ********************** End graphics setup ****************

void setup()
{
    // Setup the TFT display
  tft.init();
  tft.setRotation(0);
  tft.fillScreen(ILI9341_BLACK);
  // Setup scroll area
  setupScrollArea(TOP_FIXED_AREA, BOT_FIXED_AREA);
  
  // Setup baud rate and draw top banner
  Serial.begin(9600); // debugging

  tft.setTextColor(ILI9341_WHITE, ILI9341_BLUE);
  tft.fillRect(0,0,240,16, ILI9341_BLUE);
  tft.drawCentreString(" Serial Terminal - 9600 baud ",120,0,2);

  // Change colour for scrolling zone
  tft.setTextColor(ILI9341_WHITE, ILI9341_BLACK);

  // Zero the array
  for (byte i = 0; i<18; i++) blank[i]=0;
  
  // End TFT setup 

  radioSerial.begin(9600); // listen to radio
  Serial.print("Mesh node "); // debug output
  Serial.println(myNode);
  outputRS232();
  //createDataMessage();
  //buildOutpacket();
  println9341("APC220 Wireless Mesh");
  if (displayMeter == true) {
    drawMeter();
  }
}

void loop()
{
  rxRadio();
  NodeTimeSlot();
}

void drawMeter()
{
    int xpos;
    int ypos;
    int gap;
    int radius;
    float n;
    int reading;
    tft.setRotation(1);
    // scaling for 0-1023 to 0-20kw is to multiply by 19.55
    n = nodeValuesA0[13]; // solar value
    n = n * 19.55;
    n = (int) n;
    reading = n;
    tft.fillScreen(ILI9341_BLACK);
    xpos = 70, ypos = 40, gap = 15, radius = 84; // if make radius more than 84, then need larger fonts and then run out of memory!
    ringMeter(reading,0,16000, xpos,ypos,radius,"Solar Watts",GREEN2RED); // Draw analogue meter
}

void printlcdchar(char c)
{
    if (displayMeter == false) { // only display if not in graphics mode
      if (c == xPos > 231) {
        xPos = 0;
        yDraw = scroll_line(); // ran off end of line so do a crlf
      }
      if (c == 13) {
        xPos = 0; // carriage return, stays on the same line
      }
      if (c == 10) {
        yDraw = scroll_line(); // Line feed, It takes about 13ms to scroll 16 pixel lines
      }
      if (c > 31 && c < 128) {
        xPos += tft.drawChar(c,xPos,yDraw,2);
        blank[(18+(yStart-TOP_FIXED_AREA)/TEXT_HEIGHT)%19]=xPos; // Keep a record of line lengths
      }
    }
}  

void print9341(String s)
{
  int i;
  int j;
  j = s.length();
  for (i=0;i<j;i++)
  {
    printlcdchar(s.charAt(i));
  }  
}  

void println9341(String s)
{
  print9341(s);
  printlcdchar(13);
  printlcdchar(10); 
}  

// ##############################################################################################
// Call this function to scroll the display one text line
// ##############################################################################################
int scroll_line() {
  int yTemp = yStart; // Store the old yStart, this is where we draw the next line
  // Use the record of line lengths to optimise the rectangle size we need to erase the top line
  tft.fillRect(0,yStart,blank[(yStart-TOP_FIXED_AREA)/TEXT_HEIGHT],TEXT_HEIGHT, ILI9341_BLACK);

  // Change the top of the scroll area
  yStart+=TEXT_HEIGHT;
  // The value must wrap around as the screen memory is a circular buffer
  if (yStart >= 320 - BOT_FIXED_AREA) yStart = TOP_FIXED_AREA + (yStart - 320 + BOT_FIXED_AREA);
  // Now we can scroll the display
  scrollAddress(yStart);
  return  yTemp;
}

// ##############################################################################################
// Setup a portion of the screen for vertical scrolling
// ##############################################################################################
// We are using a hardware feature of the display, so we can only scroll in portrait orientation
void setupScrollArea(uint16_t TFA, uint16_t BFA) {
  tft.writecommand(ILI9341_VSCRDEF); // Vertical scroll definition
  tft.writedata(TFA >> 8);
  tft.writedata(TFA);     // XSTART 
  tft.writedata((320-TFA-BFA)>>8);
  tft.writedata(320-TFA-BFA);     // XSTART 
  tft.writedata(BFA >> 8);
  tft.writedata(BFA);     // XSTART 
}

// ##############################################################################################
// Setup the vertical scrolling start address
// ##############################################################################################
void scrollAddress(uint16_t VSP) {
  tft.writecommand(ILI9341_VSCRSADD); // Vertical scrolling start address
  tft.writedata(VSP>>8);
  tft.writedata(VSP);
}

 // ****************** end LCD routines *******************

// *************** Start LCD graphic routines *******************


// #########################################################################
//  Draw the meter on the screen, returns x coord of righthand side
// #########################################################################
int ringMeter(int value, int vmin, int vmax, int x, int y, int r, char *units, byte scheme)
{
  // Minimum value of r is about 52 before value text intrudes on ring
  // drawing the text first is an option
  
  x += r; y += r;   // Calculate coords of centre of ring

  int w = r / 4;    // Width of outer ring is 1/4 of radius
  
  int angle = 150;  // Half the sweep angle of meter (300 degrees)

  int text_colour = 0; // To hold the text colour

  int v = map(value, vmin, vmax, -angle, angle); // Map the value to an angle v

  byte seg = 5; // Segments are 5 degrees wide = 60 segments for 300 degrees
  byte inc = 5; // Draw segments every 5 degrees, increase to 10 for segmented ring

  // Draw colour blocks every inc degrees
  for (int i = -angle; i < angle; i += inc) {

    // Choose colour from scheme
    int colour = 0;
    switch (scheme) {
      case 0: colour = ILI9341_RED; break; // Fixed colour
      case 1: colour = ILI9341_GREEN; break; // Fixed colour
      case 2: colour = ILI9341_BLUE; break; // Fixed colour
      case 3: colour = rainbow(map(i, -angle, angle, 0, 127)); break; // Full spectrum blue to red
      case 4: colour = rainbow(map(i, -angle, angle, 63, 127)); break; // Green to red (high temperature etc)
      case 5: colour = rainbow(map(i, -angle, angle, 127, 63)); break; // Red to green (low battery etc)
      default: colour = ILI9341_BLUE; break; // Fixed colour
    }

    // Calculate pair of coordinates for segment start
    float sx = cos((i - 90) * 0.0174532925);
    float sy = sin((i - 90) * 0.0174532925);
    uint16_t x0 = sx * (r - w) + x;
    uint16_t y0 = sy * (r - w) + y;
    uint16_t x1 = sx * r + x;
    uint16_t y1 = sy * r + y;

    // Calculate pair of coordinates for segment end
    float sx2 = cos((i + seg - 90) * 0.0174532925);
    float sy2 = sin((i + seg - 90) * 0.0174532925);
    int x2 = sx2 * (r - w) + x;
    int y2 = sy2 * (r - w) + y;
    int x3 = sx2 * r + x;
    int y3 = sy2 * r + y;

    if (i < v) { // Fill in coloured segments with 2 triangles
      tft.fillTriangle(x0, y0, x1, y1, x2, y2, colour);
      tft.fillTriangle(x1, y1, x2, y2, x3, y3, colour);
      text_colour = colour; // Save the last colour drawn
    }
    else // Fill in blank segments
    {
      tft.fillTriangle(x0, y0, x1, y1, x2, y2, ILI9341_GREY);
      tft.fillTriangle(x1, y1, x2, y2, x3, y3, ILI9341_GREY);
    }
  }

  // Convert value to a string
  char buf[10];
  byte len = 4; if (value > 999) len = 5;
  dtostrf(value, len, 0, buf);

  // Set the text colour to default
  tft.setTextColor(ILI9341_WHITE, ILI9341_BLACK);
  // Uncomment next line to set the text colour to the last segment value!
  // tft.setTextColor(text_colour, ILI9341_BLACK);
  
  // Print value, if the meter is large then use big font 6, othewise use 4
  if (r > 84) tft.drawCentreString(buf, x - 5, y - 20, 6); // Value in middle
  else tft.drawCentreString(buf, x - 5, y - 20, 4); // Value in middle

  // Print units, if the meter is large then use big font 4, othewise use 2
  tft.setTextColor(ILI9341_WHITE, ILI9341_BLACK);
  if (r > 84) tft.drawCentreString(units, x, y + 30, 4); // Units display
  else tft.drawCentreString(units, x, y + 5, 2); // Units display

  // Calculate and return right hand side x coordinate
  return x + r;
}

// #########################################################################
// Return a 16 bit rainbow colour
// #########################################################################
unsigned int rainbow(byte value)
{
  // Value is expected to be in range 0-127
  // The value is converted to a spectrum colour from 0 = blue through to 127 = red

  byte red = 0; // Red is the top 5 bits of a 16 bit colour value
  byte green = 0;// Green is the middle 6 bits
  byte blue = 0; // Blue is the bottom 5 bits

  byte quadrant = value / 32;

  if (quadrant == 0) {
    blue = 31;
    green = 2 * (value % 32);
    red = 0;
  }
  if (quadrant == 1) {
    blue = 31 - (value % 32);
    green = 63;
    red = 0;
  }
  if (quadrant == 2) {
    blue = 0;
    green = 63;
    red = value % 32;
  }
  if (quadrant == 3) {
    blue = 0;
    green = 63 - 2 * (value % 32);
    red = 31;
  }
  return (red << 11) + (green << 5) + blue;
}

// #########################################################################
// Return a value in range -1 to +1 for a given phase angle in degrees
// #########################################################################
float sineWave(int phase) {
  return sin(phase * 0.0174532925);
}
 
 
// start radio routines

void txRadio(char c)
{
  // send a byte via radio, will display on the remote screen
  radioSerial.write(c);
}  

void rxRadio()
{
  byte c;
  while (radioSerial.available())
  {
    c = radioSerial.read(); // fetch the byte
    processRadioString(c); // is it a command?
  }  
  
}  

void processRadioString(char c) // converts a byte to a char, so can work with the number
{
  if (radioString.length() > 30) // string too long
  {
    radioString = ""; 
    println9341("String too long");    
  }  
  
  if ((c>31) && (c<127)) // only send printable characters and 13,, takes 2 bytes to send a value, but can use 13 as an end of line marker
  {
    radioString += c; // add to the string
    //printlcdchar(c); // print the character so can see data coming in
  }
  if (c == 13) // process the command ir there is a carriage return
  {
  //if (radioString.startsWith("SSID=")) { processSSID();} // parse commands
   if (radioString.startsWith("Time=")) { processTimeSignal();} // parse command
   if (radioString.startsWith("Data=")) { processDataMessage();} // node,sample,timestamp
   //if (radioString.startsWith("Pack=")) { processPacket();} // all data in one large packet - too large somewhere around 100 bytes starts to error
    // and now delete the radiostring
   radioString = ""; 
  }  
}  

void processTimeSignal() // pass Time=03 00000000 where 03 is the node this came from
{
 String s;
 unsigned long centraltime;
 unsigned long mytime;
 int messageFrom;
 s = stringMid(radioString,6,2);
 s = "000000" + s;
 messageFrom = (int) hexToUlong(s);
 s = radioString;
 s = stringMid(s,9,8);
 print9341("Time msg from ");
 println9341((String) messageFrom);
 if ((messageFrom <= lowestNode) && (myNode != 0)) // current time slot is less or equal than the last node update number, so refresh time, so minimum number of hops
 {
   mytime = (unsigned long) millis();
   centraltime = hexToUlong(s); // convert back to a number
   centraltime = centraltime + 250; // offset to allow for delay with transmission, determined by trial and error, if slow, then add more
   timeOffset = centraltime - mytime;
   print9341("New ");
   displayTime();
   lowestNode = messageFrom; // update with the new current lowest node number
 }  
}

 

//void millisRollover() // with rollover protection as rollover happens every 49 days. For reference baldengineer.com
//{
//  unsigned long currentMillis = millis()
//  if (((unsigned long ) (currentMillis - previousMillis) >= interval) {} // handles trigger and rollover
//}  

void refreshTime()
{
  timeCentral = ((unsigned long) millis()) + timeOffset;
}

void displayTime()
{
  refreshTime();
  //println9341ln((String) timeCentral); // print in decimal
  print9341("Time ");
  println9341(ulongToHex(timeCentral)); // print in hex
}  

void sendTime() // send unsigned long as 4 hex values
{
  txString="";
  refreshTime(); // get the current time
  //txString = String(timeCentral, HEX); // convert to hex
  //println9341ln(txString);
  //ulongToHex(timeCentral);
  print9341(ulongToHex(timeCentral));
  println9341((String) hexToUlong("000000FF"));
}  

String ulongToHex(unsigned long n) // convert an unsigned long number to a hex string 8 characters long
{
   int i;
   byte x;
   String s = "";
   for (i=0;i<8;i++) {
     x = n & 0x0000000F; // mask
     if (x<10) {
       s = (char)(x+48) + s; // 0 to 9
     }else{
       s = (char)(x+55) + s; // A to F  
     } 
     n = n >> 4; // bit shift right
   } 
  return s;
}  

unsigned long hexToUlong(String s)
{
  unsigned long n = 0; // return value
  int i; // general counter
  byte a; // byte at the position in the string
  unsigned long x = 1; // multiply by this number
  for (i=7;i>=0;i--) { // end to the start of the string
    a = (char) s.charAt(i); // get the character and convert to ascii value
    if (a<65) { // if less than 'A'
      n = n + ((a-48) * x); // convert the number
    }else{
      n = n + ((a-55) * x); // convert the letter
    }
    x = x << 4; // multipy by 16  
  }
  return n;
}  

void transmitTime() // send my local time, will converge on the lowest node number's time
{
  String nodeNumber;
  refreshTime();
  //displayTime();
  nodeNumber = stringMid(ulongToHex(myNode),7,2);
  String t="Time=";
  t = t + nodeNumber; // hex value
  t = t + " ";
  t = t + ulongToHex(timeCentral);
  radioSerial.println(t); // transmit via radio
}  

String stringLeft(String s, int i) // stringLeft("mystring",2) returns "my"
{
  String t;
  t = s.substring(0,i);
  return t;
}  

String stringMid(String s, int i, int j) // stringmid("mystring",4,3) returns "tri" (index is 1, not zero)
{
  String t;
  t = s.substring(i-1,i+j-1);
  return t;
}  
  
void processDataMessage() // Data=0312AAAABBBBBBBBcrlf where 03 is from, 12 is node (hex), AAAA is integer data, BBBBBBBB is the time stamp
{
  String s;
  unsigned long node;
  unsigned long valueA0;
  unsigned long valueA1;
  unsigned long timestamp;
  unsigned long from;
  unsigned long age;
  unsigned long previousage;
  //println9341("."); // print a dot as data comes in
  s = "000000" + stringMid(radioString,6,2); // node is 2 bytes
  from = hexToUlong(s); // get where this data came from
  s = "000000" + stringMid(radioString,8,2); // node number
  node = hexToUlong(s);
  s = "0000" + stringMid(radioString,10,4); // get the 2 bytes A0 value
  valueA0 = hexToUlong(s);
  s = "0000" + stringMid(radioString,14,4); // get the 2 bytes A1 value
  valueA1 = hexToUlong(s);
  s = stringMid(radioString,18,8);
  timestamp = hexToUlong(s);
  age = (unsigned long) (timeCentral - timestamp);
  previousage = (unsigned long) (timeCentral - nodeTimestamps[node]);
  if (age < previousage) // more recent data so update
  {
    nodeTimestamps[node] = timestamp; // update the time stamp
    nodeValuesA0[node] = (int) valueA0; // update the values
    nodeValuesA1[node] = (int) valueA1; // A1 as well
    //println9341("Update node");
    //println9341ln((String) node);
    //analogOutput(); // update the analog outputs
  }  
}  

void createDataMessage() // read A0 and A1 create data string
{
  String s;
  byte i;
  String buildMessage;
  unsigned long u;
  String myNodeString;
  updateMyValue(); // update my analog input
  print9341("My values=");
  print9341((String) nodeValuesA0[myNode]);
  print9341(",");
  println9341((String) nodeValuesA1[myNode]);
  u = (unsigned long) myNode;
  s = ulongToHex(u);
  myNodeString = "Data=" + stringMid(s,7,2); // from me
  delay(150); // for any previous message to go through
  for (i=0;i<16;i++) {
    buildMessage = myNodeString; // start building a string
    u = (unsigned long) i; // 0 to 15 - 2 bytes for the actual node number
    s = ulongToHex(u);
    buildMessage += stringMid(s,7,2);
    u = (unsigned long) nodeValuesA0[i];
    s = ulongToHex(u);
    buildMessage += stringMid(s,5,4); // data value in hex for A0
    u = (unsigned long) nodeValuesA1[i];
    s = ulongToHex(u);
    buildMessage += stringMid(s,5,4); // data value in hex for A1
    s = ulongToHex(nodeTimestamps[i]); // timestamp value for this node
    buildMessage += s;// add this
    radioSerial.println(buildMessage); // transmit via radio
    delay(150); // errors on 75, ok on 100
  }
}  

void outputRS232() // output all mesh values in one long hex string to RS232 port aaaabbbbccccdddd etc where a is node 0 value 0, b is node 0 value 1, c is node 1 value 0
{
  String buildMessage;
  byte i;
  String s;
  unsigned long u;
  buildMessage = "All values=";
  for (i=0;i<16;i++) {
    u = (unsigned long) nodeValuesA0[i];
    s = ulongToHex(u);
    buildMessage += stringMid(s,5,4); // data value in hex for A0
    u = (unsigned long) nodeValuesA1[i];
    s = ulongToHex(u);
    buildMessage += stringMid(s,5,4); // data value in hex for A1
   }   
 Serial.println(buildMessage);  
}  
  
void updateMyValue()
{
  refreshTime(); // timecentral
  int sensorValue = analogRead(A0);
  nodeValuesA0[myNode] = sensorValue;
  sensorValue = analogRead(A1);
  nodeValuesA1[myNode] = sensorValue;
  nodeTimestamps[myNode] = timeCentral; // this was updated now
}  

void NodeTimeSlot() // takes 100ms at beginning of slot so don't tx during this time.
// time slots are 4096 milliseconds - easier to work with binary maths
{
  unsigned long t;
  unsigned long remainder;
  int timeSlot = 0;
  refreshTime(); // update timeCentral
  t = timeCentral >> 12; // 4096ms counter
  t = t << 12; // round
  remainder = timeCentral - t;
  if (( remainder >=0) && (remainder < 100)) // beginning of a 4096ms time slot
  {
     //println9341("Slot ");
     timeSlot = getTimeSlot();
     print9341((String) timeSlot);
     print9341(" ");
     //printPartnerNode(); // print my partner's value
     printNode(timeSlot); // print the value of this node
     //printHeapStack(); // heap should always be less than stack
     if ( timeSlot == 0) { lowestNode = 255; } // reset if nod zero get time from the lowest node number, 0 to 15 and if 255 then has been reset
     delay(110);
     if (timeSlot == myNode) // transmit if in my time slot
     {
       println9341("Sending my data"); // all nodes transmit central time, when listening, take the lowest number and ignore others
       transmitTime();
       createDataMessage(); // send out all my local data via radio
       outputRS232(); // send out all the node values via RS232 port (eg to a PC)
       //buildPacket(); - too large, deleted this
       //printPartnerNode(); // print my partner's value
       if (displayMeter == true) {
         drawMeter(); // display one particular value on one node
       }
      }
  }  
}  

int getTimeSlot() // find out which time slot we are in
{
  int n;
  unsigned long x;
  refreshTime();
  x = timeCentral >> 12; // divide by 4096
  x = x & 0x0000000F; // mask so one of 16
  n = (int) x;
  return n;
}  

//void printPartnerNode() // can output partner node's values or do other things
//{
//  println9341("Mate ");
//  println9341((String) partnerNode);
//  println9341("=");
//  println9341((String) nodeValuesA0[partnerNode]);
//  println9341(",");
//  println9341((String) nodeValuesA1[partnerNode]);
//}  

void printNode(int x) // print current node
{
  print9341("= ");
  print9341((String) nodeValuesA0[x]);
  print9341(",");
  println9341((String) nodeValuesA1[x]);
  
}  
  
/* This function places the current value of the heap and stack pointers in the
 * variables. You can call it from any place in your code and save the data for
 * outputting or displaying later. This allows you to check at different parts of
 * your program flow.
 * The stack pointer starts at the top of RAM and grows downwards. The heap pointer
 * starts just above the static variables etc. and grows upwards. SP should always
 * be larger than HP or you'll be in big trouble! The smaller the gap, the more
 * careful you need to be. Julian Gall 6-Feb-2009.
 */
/* 
uint8_t * heapptr, * stackptr;
void check_mem() {
  stackptr = (uint8_t *)malloc(4);          // use stackptr temporarily
  heapptr = stackptr;                     // save value of heap pointer
  free(stackptr);      // free up the memory again (sets stackptr to 0)
  stackptr =  (uint8_t *)(SP);           // save value of stack pointer
}

void printHeapStack()
{
  int n;
  check_mem();
  println9341("Heap/Stack=");
  n = (int) heapptr;
  println9341((String) n);
  println9341(",");
  n = (int) stackptr;
  println9341((String) n);
}    
*/
//void analogOutput() // output on pins 3 and 5 my partner's voltages
//{
//  int x;
//  x = nodeValuesA0[partnerNode];
//  analogWrite(3, x/4); // analog inputs are 0-1023, outputs are 0-255
//  x = nodeValuesA1[partnerNode];
//  analogWrite(5, x/4);
//}
