/**************************
* Application Information *
**************************/

/*
Device: AtMega8
Clock: External Crystal @ 12MHz
hfuse: 0xC9
lfuse: 0xEF

LED row controller on port D
LED shift registers on port C
12MHz crystal on Xtal1/Xtal2
*/


/*****************
* Hardware Setup *
*****************/

/*		____________________Mega8___________________
		|Reset (PC6)		(ADC5/SCL) 	PC5|
mic2981-Pin1	|PD0 (RXD)		(ADC4/SDA) 	PC4|
mic2981-Pin2	|PD1 (TXD)			(ADC3)	PC3| hef4794-Pin15 (Enable Output)
mic2981-Pin3	|PD2 (Int0)			(ADC2)	PC2| hef4794-Pin3 (Clock)
mic2981-Pin4	|PD3 (Int2)			(ADC1)	PC1| hef4794-Pin2 (Data)
mic2981-Pin5	|PD4 (Xck/T0)			(ADC0)	PC0| hef4794-Pin1 (Latch)
+5V		|VCC					GND| Ground
Ground		|GND				       AREF|
Crystal		|PB6 (Xtal1/Tosc1)		       AVCC| +5V
Crystal		|PB7 (Xtal2/Tosc2)		(SCK)	PB5|
mic2981-Pin6	|PD5 (T1)			(MISO)	PB4|
mic2981-Pin7	|PD6 (AIN0)		(MOSI/OC2)	PB3|
mic2981-Pin8	|PD7 (AIN1)		(SS/OC1B)	PB2|
		|PB0 (ICP1)			(OC1A)	PB1|
		--------------------------------------------
*/



#define F_CPU 12000000UL

#include <avr/io.h>
#include <avr/interrupt.h>
#include <util/delay.h>
#include <avr/pgmspace.h>
#include "font5x8.h"

//The pins used to operate the shift register:
#define ShiftPort 	PORTC
#define ShiftDDR	DDRC
#define LatchPin	(1 << 0)
#define DataPin		(1 << 1)
#define ClkPin		(1 << 2)
#define OE		(1 << 3)

//The pins used to switch the row driver:
#define RowPort		PORTD
#define RowDDR		DDRD
#define RowPin0		(1 << 0)
#define RowPin1		(1 << 1)
#define RowPin2		(1 << 2)
#define RowPin3		(1 << 3)
#define RowPin4		(1 << 4)
#define RowPin5		(1 << 5)
#define RowPin6		(1 << 6)
#define RowPin7		(1 << 7)

#define ScrollSpeed	75	//How many milliseconds to pause before shifting columns left

typedef unsigned char u8;
typedef unsigned int u16;

u8 row_track = 0;

volatile u8 row_buffer[8] = { 0, 0, 0, 0, 0, 0, 0, 0 };

static const char PROGMEM Messages1[] = { "8x8 LED Display Matrix" };
static const char PROGMEM Messages2[] = { "The quick brown fox jumps over the lazy dog." };

void Delay_ms(int cnt)	//Function delays a give number of milliseconds.  Depends on F_CPU being defined
{
  while (cnt-->0) _delay_ms(1);
}

static inline void InitPorts(void)	//Function used once to initialize the ports
{
  ShiftDDR |= (LatchPin | ClkPin | DataPin | OE);	//Setup shift register control pins
  RowDDR |= (RowPin0 | RowPin1 | RowPin2 | RowPin3 | RowPin4 | RowPin5 | RowPin6 | RowPin7); //Setup row driver pins
  
  ShiftPort |= OE;
  RowPort |= RowPin0;				//Drive first row and enable shift registers

  ShiftPort |= LatchPin; 				//Set latch pin high
  ShiftPort &= ~(ClkPin | DataPin); 			//Set ClkPin and DataPin low
}

static inline void InitTimers(void)	//Function used once to set up the timer
{
  TCCR1B |= 1<<WGM12 | 1<<CS11 | 1<<CS10;		//Start timer1 in CTC mode with prescaler of 64
  TIMSK |= 1<<OCIE1A;					//Enable compare match interrupt
  OCR1A = 0x00BB;					//Set compare value for 1 mSec
  sei();						//Enable global interrupts
}

void Shift_Int(u8 shiftData)		//Function used to shift in data (Alter this when adding more than 8 columns)
{
  ShiftPort &= ~(LatchPin | ClkPin | DataPin);  	//All pins low: LatchPin low signals a write operation
  
  for (char i=0; i<8; i++)
  {
    ShiftPort &= ~ClkPin;				//Set ClkPin low

    if (shiftData & (1<<i)) ShiftPort |= DataPin;		//Set DataPin high if current bit is 1
    else ShiftPort &= ~DataPin;				//Set DataPin low if current bit is 0

    ShiftPort |= ClkPin;				//Set ClkPin high to increment shift register
    ShiftPort &= ~DataPin;				//Set DataPin low to prepare for next write
  }

  ShiftPort |= LatchPin;  				//Set LatchPin high to signal end of write operation
  ShiftPort &= ~(ClkPin | DataPin);  			//Reset ClkPin and DataPin to low
}

void each_led(void)			//A function that lights each LED one at a time
{
  for (u8 j=0; j<8; j++)
  {
    cli();
    row_buffer[j] = 0x0001;
    sei();
    Delay_ms(50);
    for (char i=0; i<8; i++)
    {
      cli();
      row_buffer[j] <<= 1;
      sei();
      Delay_ms(50);
    }
    cli();
    row_buffer[j] = 0x0000;
    sei();
  }
}

void Write_Char(u8 letter)		//Function that writes one charachter to the LED array (no scrolling)
{
  //Writes a char to the led matrix
  //TODO: Prevent non-valid characters from crashing program
  
  //letters come from font5x8[] in progmem (font5x8.h)
  letter -= 32;						//Adjust char value to match our font array indicies
  char temp;
  for (char i=0; i<5; i++)				//Read one column of char at a time
  {
    temp = pgm_read_byte((char *)((int)font5x8 + (5 * letter) + i));	//Get column from progmem
    for (char j=0; j<8; j++)						//Cycle through each bit in column
    {
      //Write bits to appropriate row_buffer location
      if (temp & (1<<j)) row_buffer[7-j] |= 1<<(4-i);
      else row_buffer[7-j] &= ~(1<<(4-i));
    }
  }
}

void shift_row_buffer_left(u8 columns)	//Function that shifts all rows a given amount to the left
{
  for (u8 buffer_index=0; buffer_index<8; buffer_index++)
  {
    row_buffer[buffer_index] <<= columns;
  }
}

void Scroll_Char (char myChar)	//Function that scrolls a character onto the display from right to left
{
  myChar -= 32;			//Adjust char value to match our font array indicies
  char temp;
  shift_row_buffer_left(1);  	//Put a blank column before each new letter.
  Delay_ms(ScrollSpeed);
  for (u8 i=0; i<5; i++)	//Read one column of char at a time
  {
    shift_row_buffer_left(1);
    temp = pgm_read_byte((char *)((int)font5x8 + (5 * myChar) + i));	//Get column from progmem
    for (u8 j=0; j<8; j++)						//Cycle through each bit in column
    {
      //Write bits to appropriate row_buffer location
      if (temp & (1<<j)) row_buffer[7-j] |= 1<<0;
      else row_buffer[7-j] &= ~(1<<0);
    }
    Delay_ms(ScrollSpeed);
  }
}
void Scroll_String(char * myString)	//Function that scrolls a given string onto the display
{
  while (*myString)			//Repeat until a null character is encountered (signals end of a string)
  {
    Scroll_Char(*myString);		//Scroll in the next character
    ++myString;				//Increment the point which tracks where we are in the string

  }
}

void Scroll_String_P(const char * myString)  //Function that reads a string out of memory and displays it
{
  while (pgm_read_byte(myString))
  {
    Scroll_Char(pgm_read_byte(myString));
    ++myString;
  }
}

void Scroll_Clear(void)			//Function that clears the display by shifting everything left
{
  for(u8 i=0; i<8; i++)
  {
    shift_row_buffer_left(1);
    Delay_ms(ScrollSpeed);
  }
}

int main(void)
{
  InitPorts();
  InitTimers();
  while(1)
  {
    Scroll_String("Hello World!");	//Write this string (not optimal because of ram usage)
    Scroll_Clear();			//Scroll left until the display is cleared
    Scroll_String_P(Messages1);		//Write a string stored in program memory (uses little ram)
    Scroll_Clear();
    Scroll_String_P(Messages2);
    Scroll_Clear();
    
    each_led();

    for (char x=33; x<128; x++)		//Writes each char in the ASCII table one at a time
    {
      Write_Char(x);
      Delay_ms(500);
    }
  }
}

ISR(TIMER1_COMPA_vect)				//Interrupt Service Routine handles the display.
{
  if(++row_track == 8) row_track = 0;		//Row tracking
  Shift_Int(row_buffer[row_track]);		//Shift in data for next row

  ShiftPort &= ~OE;				//Used to prevent ghosting (might be better to use: asm(NOP); ???)
  if (row_track == 0)				//Shut down high side controller
  {
    RowPort &= ~(1<<7);				//If Row0 is next, shudown Row7
  }
  else
  {
    RowPort &= ~(1<<(row_track-1));		//Shutdown previous row
  }

  ShiftPort |= LatchPin;			//Latch low side shift registers
  RowPort |= (1<<row_track);			//Drive high side controller
  ShiftPort |= OE;				//Used to prevent ghosting (may or may not be worth it.)
}
