
/******************************************************************************
 * Absolutelyautomation.com
 *
 * GPS based Stratum 1 NTP server, also daytime and time services
 * 
 * Daytime port 13	rfc867		tcp
 * Time 	port 37	rfc868		tcp
 * NTP 		port 123	rfc1305	udp
*******************************************************************************/

#include "ets_sys.h"
#include "osapi.h"
#include "mem.h"
#include "gpio.h"
#include "os_type.h"
#include "user_interface.h"
#include "espconn.h"
#include <include/driver/uart.h>


const char debug_udp_remote_ip[4] = { 192, 168, 2, 168 };  
const unsigned short debug_udp_remote_port = 12345; 


#define UART_RX_BUFF_SIZE	150
#define TRANS_QUEUE_LEN		20
#define CONN_QUEUE_LEN		20
#define PROC_TASK_PRIO	0
#define CONN_TASK_PRIO	1
#define TRANS_RECV_DATA_FROM_UART	1
#define CONN_CLOSE_TCP_DAYTIME	2
#define CONN_CLOSE_TCP_TIME	3


#define GPRMC_MAX_SIZE		72
const uint32 GPRMC_US_OFFSET_BYTETIME = GPRMC_MAX_SIZE *2000 ;	// Byte time @ 4800bps = 2000 us
static os_event_t trans_Queue[TRANS_QUEUE_LEN];
static os_event_t conn_Queue[CONN_QUEUE_LEN];
char NMEA_buffer[UART_RX_BUFF_SIZE] ;


static const int gpioledpin = 2;
LOCAL os_timer_t gps_watchdog;
LOCAL os_timer_t ipready_timer;
LOCAL struct espconn socket_udp_debug;
LOCAL struct espconn socket_tcp_daytime;
LOCAL struct espconn socket_tcp_time;
LOCAL struct espconn socket_udp_ntp;
static struct UartBuffer *pRxBuffer = NULL;

uint8 GPSfix;
uint8 GPS_wd_flag;
uint8 GPSsecond;
uint8 GPSminute;
uint8 GPShour;
uint16 GPSmillisecond;
uint16 GPSyear;
uint8 GPSmonth;
uint8 GPSday;
uint32 RTCcounts;

/******************************************************************************
 * FunctionName : gps_wd_func
 * Description  : Run by a software timer every 3 seconds, checks if there is no new valid NMEA frame to turn off GPSfix flag
 *                
 * Parameters   : NONE
 * Returns      : NONE
 *******************************************************************************/

void gps_wd_func(void *arg)
{
	// If no serial string received in 3 seconds, set GPS watchdog flag to 0 and GPSfix to 0
	// GPS_wd_flag should be raised every second if a valid GPS NMEA string is received

	if(GPS_wd_flag == 0)
	{
		GPSfix=0;
	}
	GPS_wd_flag=0;
	
}

/******************************************************************************
 * FunctionName : uart0_rx_intr_disable
 * Description  : disables UART0 full rx buffer and rx timeout interrupts
 *                
 * Parameters   : NONE
 * Returns      : NONE
 *******************************************************************************/

void uart0_rx_intr_disable(void)
{
    CLEAR_PERI_REG_MASK(UART_INT_ENA(0), UART_RXFIFO_FULL_INT_ENA | UART_RXFIFO_TOUT_INT_ENA);
}

/******************************************************************************
 * FunctionName : uart0_rx_intr_enable
 * Description  : enables UART0 full rx buffer and rx timeout interrupts
 *                
 * Parameters   : NONE
 * Returns      : NONE
 *******************************************************************************/

void uart0_rx_intr_enable(void)
{
    SET_PERI_REG_MASK(UART_INT_ENA(0), UART_RXFIFO_FULL_INT_ENA | UART_RXFIFO_TOUT_INT_ENA);
}

/******************************************************************************
 * FunctionName : uart0_rx_intr_handler
 * Description  : Internal used function
 *                UART0 interrupt handler, add self handle code inside. ! MODIFIED only RX interrupts !
 * Parameters   : void *para - point to ETS_UART_INTR_ATTACH's arg
 * Returns      : NONE
 *******************************************************************************/
LOCAL void
uart0_rx_intr_handler(void *para)
{
    
    if (UART_FRM_ERR_INT_ST == (READ_PERI_REG(UART_INT_ST(0)) & UART_FRM_ERR_INT_ST)) {        
        WRITE_PERI_REG(UART_INT_CLR(0), UART_FRM_ERR_INT_CLR);
    }

    if (UART_RXFIFO_FULL_INT_ST == (READ_PERI_REG(UART_INT_ST(0)) & UART_RXFIFO_FULL_INT_ST)) {
        uart0_rx_intr_disable();	
        rx_sliding_buff_enq();
    } else if (UART_RXFIFO_TOUT_INT_ST == (READ_PERI_REG(UART_INT_ST(0)) & UART_RXFIFO_TOUT_INT_ST)) {
        uart0_rx_intr_disable();
        rx_sliding_buff_enq();
    }

}

/******************************************************************************
 * FunctionName : SecondsSince1900
 * Description  : Returns input date in seconds since 00:00 GMT JAN 01 1900 
 *                			No range verification! use with care!
 * Parameters   : year > 2015, month (1-12), day (1-31), hour (0-23), minute (0-59), second (0-59)
 * Returns      : Integer 32 bits 
 *******************************************************************************/
uint32 ICACHE_FLASH_ATTR
SecondsSince1900(uint16 year, uint8 month, uint8 day, uint8 hour, uint8 minute, uint8 second)
{
	uint16 counter;
	uint16 yearcount,leapdays;
	uint32 daysfromepoch;
	uint16 daysUpToMonth[12] = { 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334 };
	uint16 daysUpToMonthLeapYear[12] = { 0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335 };

	// integer years from epoch 
	yearcount=year-1900;

	// get leapdays from epoch
	leapdays=0;
	for(counter=1900;counter<year;counter++)
	{
		if( ( ( counter%4==0) && (counter%100!=0)) || (counter%400==0 ) )         
		{
			leapdays++;
		}	
	}
	
	
	
	if( ( ( year%4==0) && (year%100!=0)) || (year%400==0 ) )         
	{
		daysfromepoch = yearcount*365 + leapdays + daysUpToMonthLeapYear[month-1]+(day-1);
	}
	else
	{
		daysfromepoch = yearcount*365 + leapdays + daysUpToMonth[month-1]+(day-1);
	}

	return( (daysfromepoch*86400)+(hour*3600)+(minute*60)+second );
}

/******************************************************************************
 * FunctionName : Uart_Buf_Init
 * Description  : tx buffer enqueue: fill a first linked buffer
 * Parameters   : char *pdata - data point  to be enqueue
 * Returns      : NONE
 *******************************************************************************/
struct UartBuffer *ICACHE_FLASH_ATTR
Uart_Buf_Init(uint32 buf_size)
{
    uint32 heap_size = system_get_free_heap_size();

    if(buf_size > 65535) { // now not support
        os_printf("no buf for uart\n\r");
        return NULL;
    }
    if (heap_size <= buf_size) {
        os_printf("no buf for uart\n\r");
        return NULL;
    } else {
        os_printf("test heap size: %d\n\r", heap_size);
        struct UartBuffer *pBuff = (struct UartBuffer *)os_malloc((uint32)sizeof(struct UartBuffer));
        pBuff->UartBuffSize = (uint16)buf_size; // THIS OK
        pBuff->pUartBuff = (uint8 *)os_malloc(pBuff->UartBuffSize);
        pBuff->pInPos = pBuff->pUartBuff;
        pBuff->pOutPos = pBuff->pUartBuff;
        pBuff->Space = pBuff->UartBuffSize;
        pBuff->BuffState = OK;
        pBuff->nextBuff = NULL;
        //        pBuff->TcpControl = RUN;
        return pBuff;
    }
}

/******************************************************************************
 * FunctionName : rx_sliding_buff_enq
 * Description  : enables UART0 full rx buffer and rx timeout interrupts
 *                
 * Parameters   : NONE
 * Returns      : NONE
 *******************************************************************************/

rx_sliding_buff_enq(void)
{
    uint8 fifo_len = 0;
    uint8 str_len = 0;
    uint8 str_len2 = 0;
    ETSParam par = 0;
	

    uint8* tail = (pRxBuffer->pUartBuff + pRxBuffer->UartBuffSize);

    fifo_len = (READ_PERI_REG(UART_STATUS(UART0)) >> UART_RXFIFO_CNT_S)&UART_RXFIFO_CNT;

        
    while (fifo_len--) {
        *(pRxBuffer->pInPos++) = READ_PERI_REG(UART_FIFO(UART0)) & 0xFF;
        if (pRxBuffer->pInPos == tail) {
            pRxBuffer->pInPos = pRxBuffer->pUartBuff;
        }
    }

        // Arranging characteres in a "sliding window" buffer
	str_len = tail - (pRxBuffer->pInPos);
	os_memcpy(NMEA_buffer, pRxBuffer->pInPos, str_len);
	str_len2 = (pRxBuffer->pInPos)-(pRxBuffer->pUartBuff);
	os_memcpy(NMEA_buffer+str_len, pRxBuffer->pUartBuff, str_len2);

    if(system_os_post(PROC_TASK_PRIO, (ETSSignal)TRANS_RECV_DATA_FROM_UART, par) != TRUE) {
        os_printf("POST proc data from UART0 fail! !\n\r");
    }
    WRITE_PERI_REG(UART_INT_CLR(0), UART_RXFIFO_FULL_INT_CLR | UART_RXFIFO_TOUT_INT_CLR);
    uart0_rx_intr_enable();
}

/******************************************************************************
 * FunctionName : uart0_int_setup
 * Description  : Simpler and flexible function to setup UART0 interrupts
 * Parameters  : interrupt bit, additional parameter
 *				   additional parameter is only used for: RXFIFO_FULL_INT, RXFIFO_TOUT_INT, TXFIFO_EMPTY_INT
 * Returns      : NONE
 *******************************************************************************/

void ICACHE_FLASH_ATTR 

uart0_int_setup(unsigned short intbit,unsigned char parameter)
{
    unsigned long regvalue;
    
    SET_PERI_REG_MASK(UART_INT_ENA(0), intbit );
    if( intbit == UART_RXFIFO_FULL_INT_ENA )
    {
	regvalue=READ_PERI_REG(UART_CONF1(0)) | ( (100 & parameter)<<UART_RXFIFO_FULL_THRHD_S);
	WRITE_PERI_REG(UART_CONF1(0),regvalue);
    }

    if( intbit == UART_RXFIFO_TOUT_INT_ENA)
    {
	//SET_PERI_REG_MASK(UART_CONF1(0), UART_RX_TOUT_EN  );
	regvalue=READ_PERI_REG(UART_CONF1(0)) | UART_RX_TOUT_EN | ( (0x02 & parameter)<<UART_RX_TOUT_THRHD_S);
	WRITE_PERI_REG(UART_CONF1(0),regvalue);
    }

    if( intbit == UART_TXFIFO_EMPTY_INT_ST)
    {
	regvalue=READ_PERI_REG(UART_CONF1(0)) | ( (0x10 & parameter)<<UART_TXFIFO_EMPTY_THRHD_S);
	WRITE_PERI_REG(UART_CONF1(0),regvalue);
    }
    //clear rx and tx fifo,not ready
   SET_PERI_REG_MASK(UART_CONF0(0), UART_RXFIFO_RST | UART_TXFIFO_RST);    //RESET FIFO
   CLEAR_PERI_REG_MASK(UART_CONF0(0), UART_RXFIFO_RST | UART_TXFIFO_RST);

}

/******************************************************************************
 * FunctionName : tcp_daytime_sent_cb
 * Description  : Call back function after a daytime message (port 13) delivered, Sends a message to a task for tcp comm closing
 * Parameters  : 
 *				   
 * Returns      : NONE
 *******************************************************************************/

void ICACHE_FLASH_ATTR
tcp_daytime_sent_cb(void *arg)
{
	ETSParam par = 0;
	if(system_os_post(CONN_TASK_PRIO, (ETSSignal)CONN_CLOSE_TCP_DAYTIME, par) != TRUE) 
		{
        		os_printf("Close TCP Daytime fail! !\n\r");
		}
}

/******************************************************************************
 * FunctionName : tcp_time_sent_cb
 * Description  : Call back function after a time message (port 37) delivered, Sends a message to a task for tcp comm closing
 * Parameters  : 
 *				   
 * Returns      : NONE
 *******************************************************************************/

void ICACHE_FLASH_ATTR
tcp_time_sent_cb(void *arg)
{
	ETSParam par = 0;
	if(system_os_post(CONN_TASK_PRIO, (ETSSignal)CONN_CLOSE_TCP_TIME, par) != TRUE) 
		{
        		os_printf("Close TCP Time fail! !\n\r");
		}
}

/******************************************************************************
 * FunctionName : udp_ntp_recvcb
 * Description  : Call back function after a NTP message (port 123) received, processes the packet and send the answer
 * Parameters  : 
 *				   
 * Returns      : NONE
 *******************************************************************************/

void ICACHE_FLASH_ATTR
udp_ntp_recvcb(void *arg, char *pdata, unsigned short len)
{
	struct espconn *pesp_conn = arg;
	remot_info *premot = NULL;
	char ntp_packet[48];
	uint32 timetmp;
	uint32 fractmp;
	uint32 useconds;

	if (espconn_get_connection_info(pesp_conn,&premot,0) == ESPCONN_OK)
		{

			if(len>47)
			{
				// read incoming ntp packet and store localy
				os_memcpy(&ntp_packet[0], pdata,48 );				
				pesp_conn->proto.tcp->remote_port = premot->remote_port;
				pesp_conn->proto.tcp->remote_ip[0] = premot->remote_ip[0];
				pesp_conn->proto.tcp->remote_ip[1] = premot->remote_ip[1];
				pesp_conn->proto.tcp->remote_ip[2] = premot->remote_ip[2];
				pesp_conn->proto.tcp->remote_ip[3] = premot->remote_ip[3];				

				//NTP values!!!
				
				if(GPSfix==1)
				{
					ntp_packet[0]=0x1C;	// Leap 0x0, Version 0x3, Mode 0x4
				}
				else
				{
					ntp_packet[0]=0xDC;	// Leap 0x3, Version 0x3, Mode 0x4
				}

				
				ntp_packet[1]=0x01;	// Stratum 0x1
				ntp_packet[2]=0x03;	// Poll  0x3 (invalid)
				ntp_packet[3]=0xE3;	// Precision  0,00000 sec?

				ntp_packet[4]=0;		// Root Delay?
				ntp_packet[5]=0;		// Root Delay?
				ntp_packet[6]=0;		// Root Delay?
				ntp_packet[7]=0;		// Root Delay?

				ntp_packet[8]=0;		// Root Dispersion?
				ntp_packet[9]=0;		// Root Dispersion?
				ntp_packet[10]=0;	// Root Dispersion?
				ntp_packet[11]=0;	// Root Dispersion?
				
				ntp_packet[12]=0x47;	// Reference ID, "G"
				ntp_packet[13]=0x50;	// Reference ID, "P"
				ntp_packet[14]=0x53;	// Reference ID, "S"
				ntp_packet[15]=0x00;	// Reference ID, 0x00
				
				timetmp=SecondsSince1900(GPSyear, GPSmonth, GPSday, GPShour, GPSminute, GPSsecond);

				ntp_packet[16]=(char)(0x000000FF & timetmp >> 24 );	// Reference timestamp seconds
				ntp_packet[17]=(char)(0x000000FF & timetmp >> 16 );	// Reference timestamp seconds
				ntp_packet[18]=(char)(0x000000FF & timetmp >> 8 );	// Reference timestamp seconds
				ntp_packet[19]=(char)(0x000000FF & timetmp);	// Reference timestamp seconds

				ntp_packet[20]=0x0;	// Reference timestamp fraction
				ntp_packet[21]=0x0;	// Reference timestamp fraction
				ntp_packet[22]=0x0;	// Reference timestamp fraction
				ntp_packet[23]=0x0;	// Reference timestamp fraction
				
				// Origin timestamp [24]..[27] seconds , [28]..[31] fraction				

				ntp_packet[24]=ntp_packet[40];
				ntp_packet[25]=ntp_packet[41];
				ntp_packet[26]=ntp_packet[42];
				ntp_packet[27]=ntp_packet[43];

				ntp_packet[28]=ntp_packet[44];
				ntp_packet[29]=ntp_packet[45];
				ntp_packet[30]=ntp_packet[46];
				ntp_packet[31]=ntp_packet[47];


				timetmp=SecondsSince1900(GPSyear, GPSmonth, GPSday, GPShour, GPSminute, GPSsecond);
				fractmp=system_get_time();

				if( fractmp >= RTCcounts )
				{
					useconds=fractmp - RTCcounts;
				}
				else
				{
					useconds= (0xFFFFFFFF - RTCcounts)+fractmp;
				}

				useconds=useconds+GPRMC_US_OFFSET_BYTETIME;

				ntp_packet[32]=(char)(0x000000FF & timetmp >> 24 );	// Receive timestamp seconds
				ntp_packet[33]=(char)(0x000000FF & timetmp >> 16 );	// Receive timestamp seconds
				ntp_packet[34]=(char)(0x000000FF & timetmp >> 8 );	// Receive timestamp seconds
				ntp_packet[35]=(char)(0x000000FF & timetmp);	// Receive timestamp seconds
				
				ntp_packet[36]=(char)(0x000000FF & useconds >> 24 );	// Receive timestamp fraction
				ntp_packet[37]=(char)(0x000000FF & useconds >> 16 );	// Receive timestamp fraction
				ntp_packet[38]=(char)(0x000000FF & useconds >> 8 );	// Receive timestamp fraction
				ntp_packet[39]=(char)(0x000000FF & useconds);	// Receive timestamp fraction

				timetmp=SecondsSince1900(GPSyear, GPSmonth, GPSday, GPShour, GPSminute, GPSsecond);
				fractmp=system_get_time();

				if( fractmp >= RTCcounts )
				{
					useconds=fractmp - RTCcounts;
				}
				else
				{
					useconds= (0xFFFFFFFF - RTCcounts)+fractmp;
				}
				
				useconds=useconds+GPRMC_US_OFFSET_BYTETIME;

				ntp_packet[40]=(char)(0x000000FF & timetmp >> 24 );	// Transmit timestamp seconds
				ntp_packet[41]=(char)(0x000000FF & timetmp >> 16 );	// Transmit timestamp seconds
				ntp_packet[42]=(char)(0x000000FF & timetmp >> 8 );	// Transmit timestamp seconds
				ntp_packet[43]=(char)(0x000000FF & timetmp);	// Transmit timestamp seconds

				ntp_packet[44]=(char)(0x000000FF & useconds >> 24 );	// Transmit timestamp fraction
				ntp_packet[45]=(char)(0x000000FF & useconds >> 16 );	// Transmit timestamp fraction
				ntp_packet[46]=(char)(0x000000FF & useconds >> 8 );	// Transmit timestamp fraction
				ntp_packet[47]=(char)(0x000000FF & useconds);	// Transmit timestamp fraction

				espconn_sent(pesp_conn, &ntp_packet[0], 48);
			}
	
			
			//espconn_sent(pesp_conn, pusrdata, os_strlen(pusrdata));
			//os_printf("UDP data Ntp received %d bytes %d.%d.%d.%d : %d !!\n\r",len,premot->remote_ip[0],premot->remote_ip[1],premot->remote_ip[2],premot->remote_ip[3],premot->remote_port);
		}

	
}

/******************************************************************************
 * FunctionName : tcp_time_listen
 * Description  : Socket listener function for time service (port 37), processes the packet and send the answer
 * Parameters  : 
 *				   
 * Returns      : NONE
 *******************************************************************************/

static void ICACHE_FLASH_ATTR
tcp_time_listen(void *arg)
{
	
	char packet[4];
	uint32 tmptime;

	tmptime=SecondsSince1900(GPSyear,GPSmonth,GPSday,GPShour,GPSminute,GPSsecond);
	
	packet[0]=(char)(0x000000FF & tmptime >> 24 );
	packet[1]=(char)(0x000000FF & tmptime >> 16 );
	packet[2]=(char)(0x000000FF & tmptime >> 8 );
	packet[3]=(char)(0x000000FF & tmptime);

	espconn_sent( &socket_tcp_time, &packet[0], 4 );

	//os_printf("Connection on port 37");
	return;
}

/******************************************************************************
 * FunctionName : tcp_daytime_listen
 * Description  : Socket listener function for daytime service (port 13), processes the packet and send the answer
 * Parameters  : 
 *				   
 * Returns      : NONE
 *******************************************************************************/

static void ICACHE_FLASH_ATTR
tcp_daytime_listen(void *arg)
{
	char message[60];
	char tmpmesg[60];
	char smonth[20];
	char wday[20];
	

	switch(GPSmonth) 
	{

   		case 1  :
			os_sprintf(&smonth[0],"January");
			break;	
		case 2  :
      			os_sprintf(&smonth[0],"February");
			break;
		case 3  :
      			os_sprintf(&smonth[0],"March");
			break;
   		case 4  :
      			os_sprintf(&smonth[0],"April");
			break;
		case 5  :
      			os_sprintf(&smonth[0],"May");
			break;
		case 6  :
      			os_sprintf(&smonth[0],"June");
			break;
		case 7  :
      			os_sprintf(&smonth[0],"July");
			break;
		case 8  :
      			os_sprintf(&smonth[0],"August");
			break;
		case 9  :
      			os_sprintf(&smonth[0],"September");
			break;
		case 10  :
      			os_sprintf(&smonth[0],"October");
			break;
		case 11  :
      			os_sprintf(&smonth[0],"November");
			break;
		case 12  :
      			os_sprintf(&smonth[0],"December");
			break;


	}

	// Tomohiko Sakamoto's Algorithm 
	sint16 ye, mo, da, dw; 
	static sint8 t[] = {0, 3, 2, 5, 0, 3, 5, 1, 4, 6, 2, 4};
	ye=GPSyear;
	mo=GPSmonth;
	da=GPSday;

   	 ye -= mo < 3;
    	dw=(ye + ye/4 - ye/100 + ye/400 + t[mo-1] + da) % 7;    	
	switch(dw) 
	{

   		case 0  :
			os_sprintf(&wday[0],"Sunday");
			break;	
		case 1  :
      			os_sprintf(&wday[0],"Monday");
			break;
		case 2  :
      			os_sprintf(&wday[0],"Tuesday");
			break;
   		case 3  :
      			os_sprintf(&wday[0],"Wednesday");
			break;
		case 4  :
      			os_sprintf(&wday[0],"Thursday");
			break;
		case 5  :
      			os_sprintf(&wday[0],"Friday");
			break;
		case 6  :
      			os_sprintf(&wday[0],"Saturday");
			break;		

	}
	
	// os_sprintf() doesn't like more than one string argument!! , os_strcat, strcat throw compiler errors
	// so memcpy is your friend!

	os_memcpy(message, wday,(uint8)os_strlen(&wday[0]));				
	os_bzero(message+os_strlen(&wday[0]),(uint8) 1 );									
	os_sprintf(&tmpmesg[0],", %s %d, %d %d:%d:%d UTC\n\r",smonth,GPSday,GPSyear,GPShour,GPSminute,GPSsecond);
	os_memcpy(message+os_strlen(&message[0]), tmpmesg,(uint8)os_strlen(&tmpmesg[0]));				

	espconn_sent( &socket_tcp_daytime, &message[0], os_strlen(&message[0]) );
	//os_printf("Connection on port 13");
	return;
}

/******************************************************************************
 * FunctionName : debug_udp_sent_cb
 * Description  : Callback for debug purposes
 * Parameters  : NONE
 * Returns      : NONE
 *******************************************************************************/

void ICACHE_FLASH_ATTR
debug_udp_sent_cb(void *arg)
{
	struct espconn *pespconn = arg;
	os_printf("Debug UDP socket data  sent\n");
}


/******************************************************************************
 * FunctionName : create_sockets
 * Description  : Create sockets after got wireless connection and IP Address
 * Parameters  : NONE
 * Returns      : NONE
 *******************************************************************************/

void ICACHE_FLASH_ATTR 
create_sockets(void)
{
    struct ip_info ipconfig;

   //disarm timer first
    os_timer_disarm(&ipready_timer);

   //get ip info of ESP8266 station
    wifi_get_ip_info(STATION_IF, &ipconfig);

    if (wifi_station_get_connect_status() == STATION_GOT_IP && ipconfig.ip.addr != 0) 
   {
		
		//******************************************************************************************************
		// 		debug socket
		//*******************************************************************************************************

		socket_udp_debug.type = ESPCONN_UDP;
		socket_udp_debug.proto.udp = (esp_udp *)os_zalloc(sizeof(esp_udp));
		socket_udp_debug.proto.udp->local_port = espconn_port();  // set a available  port		
		socket_udp_debug.proto.udp->remote_port = debug_udp_remote_port; 
      		os_memcpy(socket_udp_debug.proto.udp->remote_ip, debug_udp_remote_ip, 4); // ESP8266 udp remote IP
            		
		espconn_regist_sentcb(&socket_udp_debug, debug_udp_sent_cb); // register a udp packet sent callback      
      		espconn_create(&socket_udp_debug);   // create udp

		//******************************************************************************************************
		// 		daytime service tcp socket	port 13
		//*******************************************************************************************************
		
		
		socket_tcp_daytime.type = ESPCONN_TCP;
		socket_tcp_daytime.state = ESPCONN_NONE;
		socket_tcp_daytime.proto.tcp = (esp_tcp *)os_zalloc((uint32)sizeof(esp_tcp));
		socket_tcp_daytime.proto.tcp->local_port = 13;      // daytime service  port
		espconn_regist_connectcb(&socket_tcp_daytime, tcp_daytime_listen);
		espconn_regist_sentcb(&socket_tcp_daytime, tcp_daytime_sent_cb);
		espconn_accept(&socket_tcp_daytime);
		espconn_regist_time(&socket_tcp_daytime, 30, 0);

		//******************************************************************************************************
		// 		time service tcp socket		port 37
		//*******************************************************************************************************
		
		socket_tcp_time.type = ESPCONN_TCP;
		socket_tcp_time.state = ESPCONN_NONE;
		socket_tcp_time.proto.tcp = (esp_tcp *)os_zalloc((uint32)sizeof(esp_tcp));
		socket_tcp_time.proto.tcp->local_port = 37;      // time service  port
		espconn_regist_connectcb(&socket_tcp_time, tcp_time_listen);
		espconn_regist_sentcb(&socket_tcp_time, tcp_time_sent_cb);
		espconn_accept(&socket_tcp_time);
		espconn_regist_time(&socket_tcp_time, 30, 0);

		//******************************************************************************************************
		// 		ntp service udp socket		port 123
		//*******************************************************************************************************
		
		socket_udp_ntp.type = ESPCONN_UDP;
		socket_udp_ntp.proto.udp = (esp_udp *)os_zalloc(sizeof(esp_udp));
		socket_udp_ntp.proto.udp->local_port = 123; 		
		espconn_create(&socket_udp_ntp);
		espconn_regist_recvcb(&socket_udp_ntp, udp_ntp_recvcb);

    } 
   else 
   {
        if ((wifi_station_get_connect_status() == STATION_WRONG_PASSWORD ||
                wifi_station_get_connect_status() == STATION_NO_AP_FOUND ||
                wifi_station_get_connect_status() == STATION_CONNECT_FAIL)) 
        {
         os_printf("connect fail !!! \r\n");
        } 
      else 
      {
           //re-arm timer to check ip again
            os_timer_setfn(&ipready_timer, (os_timer_func_t *)create_sockets, NULL);
            os_timer_arm(&ipready_timer, 100, 0);
        }
    }
}


/******************************************************************************
 * FunctionName : uart0_baud_setup
 * Description  : Simpler function to setup UART0, enables RX and TX channel ( NO flow control!! )               
 * Parameters  : baudrate, databits, parity enabled/disabled, parity type, stopbits
 *				   If parity is disabled, parity type doesn't matters
 * Returns      : NONE
 *******************************************************************************/
void ICACHE_FLASH_ATTR 

uart0_baud_setup(unsigned int baudrate,unsigned char databits,
unsigned char parenabled, unsigned char partype,unsigned char stopbits)
{
    UartDevice	UartConf;
    
    /* Available baudrate options:
    BIT_RATE_300
    BIT_RATE_600
    BIT_RATE_1200
    BIT_RATE_2400
    BIT_RATE_4800
    BIT_RATE_9600
    BIT_RATE_19200
    BIT_RATE_38400
    BIT_RATE_57600
    BIT_RATE_74880
    BIT_RATE_115200
    BIT_RATE_230400
    BIT_RATE_460800
    BIT_RATE_921600
    BIT_RATE_1843200
    BIT_RATE_3686400 */

    /* Available databits options
    FIVE_BITS
    SIX_BITS
    SEVEN_BITS
    EIGHT_BITS */

    /* Available parenabled options
    STICK_PARITY_DIS
    STICK_PARITY_EN */

    /* Available partype options
    ODD_BITS
    EVEN_BITS */

    /* Available stop bits options
    ONE_STOP_BIT
    ONE_HALF_STOP_BIT
    TWO_STOP_BIT */
    
    // TX0 pin enabled
    PIN_FUNC_SELECT(PERIPHS_IO_MUX_U0TXD_U, FUNC_U0TXD);
    // pullup disabled on TX0 pin
    PIN_PULLUP_DIS(PERIPHS_IO_MUX_U0TXD_U);
    //U0RXD default behavior RX0
    
    // Pullup disabled on RX0 pin
    PIN_PULLUP_DIS(PERIPHS_IO_MUX_U0RXD_U);
    // U0RXD default function UART0 RX

    // baudrate set
    uart_div_modify(0, (uint16)(UART_CLK_FREQ / (uint32)(baudrate) ) );
    
    // Temporal structure to hold values
    UartConf.data_bits=databits;
    UartConf.exist_parity=parenabled;
    UartConf.parity=partype;
    UartConf.stop_bits=stopbits;
    
    // Writing values to UART0 CONF0 register
    WRITE_PERI_REG(UART_CONF0(0), (uint32)UartConf.exist_parity
            | (uint32)UartConf.parity
            | (((uint8)UartConf.stop_bits) << UART_STOP_BIT_NUM_S)
            | (((uint8)UartConf.data_bits) << UART_BIT_NUM_S));

}

/******************************************************************************
 * FunctionName : connection_data_task
 * Description  : Task for closing tcp comm sockets
 * Parameters  : An event descriptor for the socket to be closed
 *				   
 * Returns      : NONE
 *******************************************************************************/

static void ICACHE_FLASH_ATTR 
connection_data_task(os_event_t *events)
{

	if(events == NULL) {
		return;
	}

	switch(events->sig) {

	case CONN_CLOSE_TCP_DAYTIME:   
		espconn_disconnect(&socket_tcp_daytime);	
		break;
	
	case CONN_CLOSE_TCP_TIME:   
		espconn_disconnect(&socket_tcp_time);	
		break;
	}


}

/******************************************************************************
 * FunctionName : process_data_task
 * Description  : Task for parsing a valid NMEA $GPRMC sentence that contains timing information.
 * Parameters  : 
 *				   
 * Returns      : NONE
 *******************************************************************************/

static void ICACHE_FLASH_ATTR 
process_data_task(os_event_t *events)
{
	static uint8 state = 0;
	int32 ret = 0;
	uint32 data_len = 0;
	uint8 sectmp;
	uint8 mintmp;
	uint8 hourtmp;
	uint8 monthtmp;
	uint8 daytmp;
	uint16 msectmp;
	uint16 yeartmp;

	unsigned char GPRMC_time[11] ;
	unsigned char GPRMC_date[7] ;
	unsigned char GPRMC_fix[2] ;

	unsigned short sizefound;
	unsigned short strsize;
	unsigned char count;
	char * gprmc_found;
	char * sep1_found;
	char * sep2_found;

	if(events == NULL) {
		return;
	}

	switch(events->sig) {		
	case TRANS_RECV_DATA_FROM_UART: 

		// char *os_strstr(const char *haystack, const char *needle);
		gprmc_found =(char *) os_strstr(NMEA_buffer, "$GPRMC");
		if( gprmc_found != NULL  )
		{	
			sizefound=(gprmc_found - NMEA_buffer);
			// check if found the whole string!! not just a piece
			if( sizefound+ GPRMC_MAX_SIZE < UART_RX_BUFF_SIZE)
				{					
					
					//$GPRMC,201705.000,A,0000.0000,N,00000.0000,W,1.10,265.50,120816,,,A*79	
					
					// Parsing time using "," characters #1 and # 2
					// **********************************************************
					sep1_found =(char *) os_strstr(gprmc_found, "," )+1 ;
					sep2_found =(char *) os_strstr(sep1_found, "," ) ;					
					strsize = sep2_found - sep1_found;					
					os_memcpy(GPRMC_time, sep1_found,(uint8)strsize);				
					os_bzero(GPRMC_time+(char)strsize,(uint8) 1 );									
					//os_printf("GPS Time: %s\r\n",GPRMC_time);
																

					sectmp=((uint8)GPRMC_time[4]-(uint8)0x30)*10;
					sectmp=sectmp+((uint8)GPRMC_time[5]-(uint8)0x30);
					//os_printf("GPS Second: %d\r\n",sectmp);
					
					mintmp=((uint8)GPRMC_time[2]-(uint8)0x30)*10;
					mintmp=mintmp+((uint8)GPRMC_time[3]-(uint8)0x30);
					//os_printf("GPS Minute: %d\r\n",mintmp);


					hourtmp=((uint8)GPRMC_time[0]-(uint8)0x30)*10;
					hourtmp=hourtmp+((uint8)GPRMC_time[1]-(uint8)0x30);
					//os_printf("GPS Minute: %d\r\n",hourtmp);

					// Extract and convert milliseconds ( some GPS doesn' )
					if( strsize > 7 )
					{
						msectmp=((uint16)GPRMC_time[7]-(uint16)0x30)*100;
						if(strsize>8)
						{
							msectmp=msectmp+( (uint16)GPRMC_time[8]-(uint16)0x30)*10;
							if(strsize>9)
							{
								msectmp=msectmp+( (uint16)GPRMC_time[9]-(uint16)0x30);
							}

						}
					}
					else
					{
						msectmp=0;
					}					
					//os_printf("GPS milliSecond: %d\r\n",msectmp);


					// Parsing date using "," characters #9 and # 10
					// **********************************************
					
					sep1_found=gprmc_found;
					for(count=0 ; count < 9 ; count ++)
					{
						sep2_found =(char *) os_strstr(sep1_found, "," )+1 ;
						sep1_found=sep2_found;
					}
					
					sep2_found =(char *) os_strstr(sep1_found, "," ) ;
					strsize = sep2_found - sep1_found;										
					os_memcpy(GPRMC_date, sep1_found,(uint8)strsize);				
					os_bzero(GPRMC_date+(char)strsize,(uint8) 1 );									
					//os_printf("GPS Date: %s\r\n",GPRMC_date);

					yeartmp=((uint16)GPRMC_date[4]-(uint16)0x30)*10;
					yeartmp=yeartmp+((uint8)GPRMC_date[5]-(uint16)0x30);
					yeartmp=yeartmp+2000;				
					//os_printf("GPS Year: %d\r\n",yeartmp);

					monthtmp=((uint8)GPRMC_date[2]-(uint8)0x30)*10;
					monthtmp=monthtmp+((uint8)GPRMC_date[3]-(uint8)0x30);
					//os_printf("GPS Month: %d\r\n",monthtmp);

					daytmp=((uint8)GPRMC_date[0]-(uint8)0x30)*10;
					daytmp=daytmp+((uint8)GPRMC_date[1]-(uint8)0x30);
					//os_printf("GPS Day: %d\r\n",daytmp);

					// Parsing satellite fix "," characters #2 and # 3
					// **********************************************
					sep1_found =(char *) os_strstr(gprmc_found, "," )+1 ;
					sep2_found = sep1_found;
					sep1_found =(char *) os_strstr(sep2_found, "," )+1 ;
					os_memcpy(GPRMC_fix, sep1_found,(uint8) 1 );				
					os_bzero(GPRMC_fix+(char) 1 ,(uint8) 1 );									
					//os_printf("GPS Fix: %d\r\n",GPSfix);
					

					//if seconds changed update global vars and get uP ticks
					if(GPSsecond != sectmp)	
					{	
						// **** update time variables should be Atomic!!! *****

						// Serial string received, rise watchdog flag
						GPS_wd_flag=1;

						if( os_strcmp(GPRMC_fix, "A") == 0 )
						{
							GPSfix=1;
						}
						else
						{
							GPSfix=0;
						}

						GPSsecond=sectmp;
						GPSminute=mintmp;
						GPShour=hourtmp;
						GPSmillisecond=msectmp;
						GPSyear=yeartmp;
						GPSmonth=monthtmp;
						GPSday=daytmp;						
						RTCcounts=system_get_time();

    						// toggle led
  						if (GPIO_REG_READ(GPIO_OUT_ADDRESS) & (1 << gpioledpin))
  						{   
    							gpio_output_set(0, (1 << gpioledpin), 0, 0);
  						}
  						else
  						{
							gpio_output_set((1 << gpioledpin), 0, 0, 0);    
						}

						//os_printf("RTC counts:%d\r\n",RTCcounts);
						//os_printf("%d:%d:%d %d-%d-%d\r\n",GPShour,GPSminute,GPSsecond,GPSday,GPSmonth,GPSyear);
					}

				}

		}

		// espconn_send(&socket_udp_debug, NMEA_BUFFER[0], os_strlen(NMEA_BUFFER[0]));				
		//os_printf("Buff Data: %s\r\n",NMEA_buffer);
		// os_printf("Str len: %d\r\n",os_strlen(NMEA_BUFFER));

		break;
	default:
		os_printf("sig %d\r\n",events->sig);
		break;
	}
}

/******************************************************************************
 * FunctionName : user_init
 * Description  : Hardware and other initializations that must be done after poweron
 * Parameters  : 
 *				   
 * Returns      : NONE
 *******************************************************************************/

void ICACHE_FLASH_ATTR user_init()
{
	// init gpio susbytem
	gpio_init();
	

	// GPIO PINS SETUP
	// ******************
	// configure GPIO2_U  to be GPIO2, disable pull-up, set as output
	PIN_FUNC_SELECT(PERIPHS_IO_MUX_GPIO2_U, FUNC_GPIO2);
	PIN_PULLUP_DIS(PERIPHS_IO_MUX_GPIO2_U);
	gpio_output_set(0, 0, (1 << gpioledpin), 0);

	// UART0 SETUP
	// *****************
	uart0_baud_setup(BIT_RATE_4800,EIGHT_BITS,STICK_PARITY_DIS,ODD_BITS,ONE_STOP_BIT);        	
		

	//clear all interrupts
	WRITE_PERI_REG(UART_INT_ENA(0), 0x0000);
	// Set interrupt when rx fifo over threshold, and set threshold 35
	uart0_int_setup(UART_RXFIFO_FULL_INT_ENA,40);
	// Set interrupt when rx timeout 10 chars
	uart0_int_setup(UART_RXFIFO_TOUT_INT_ENA,3);
	// Attach int service routine function
	ETS_UART_INTR_ATTACH(uart0_rx_intr_handler,NULL);
	// Enable gobal UART interupts
	ETS_UART_INTR_ENABLE();
		
 	pRxBuffer = Uart_Buf_Init(UART_RX_BUFF_SIZE);

	// WiFi Station setup
	// works as Client connected to a WiFi AP, please modify values in user_config.h
	// ******************
	char ssid[32] = SSID;
	char password[64] = SSID_PASSWORD;
	char hostname[32] = HOST_NAME;
	struct station_config stationConf;
	//Set station mode
	wifi_set_opmode( 0x1 );
	//Set ap settings
	os_memcpy(&stationConf.ssid, ssid, 32);
	os_memcpy(&stationConf.password, password, 64);	
	wifi_station_set_hostname(hostname);
	wifi_station_set_config(&stationConf);

	// Socket setup
	//set a timer to check whether got ip from router succeed or not. If suceed create sockets
	os_timer_disarm(&ipready_timer);
	os_timer_setfn(&ipready_timer, (os_timer_func_t *)create_sockets, NULL);
	os_timer_arm(&ipready_timer, 100, 0);

	// create a task that processes UART0 received data	
	system_os_task(process_data_task, PROC_TASK_PRIO, trans_Queue, TRANS_QUEUE_LEN);  

	// create a task that closes socket communications
	system_os_task(connection_data_task, CONN_TASK_PRIO, conn_Queue, CONN_QUEUE_LEN);  
	

	// setup GPS watchdog (3000ms, repeating) if no serial string received in 3s
	os_timer_setfn(&gps_watchdog, (os_timer_func_t *)gps_wd_func, NULL);
	os_timer_arm(&gps_watchdog, 3000, 1);

}
