/*
 * Servo Switch (SESW) - Test Program
 * For more details visit: http://www.oomlout.com/serb 
 * 
 * Behaviour: The default behaviour is to turn the switch on then wait five
 *            seconds before turning the switch off, and waiting five 
 *            seconds. This will repeat. 
 *            There are also different demo modes entered by pulling the
 *            corresponding pin LOW:
 *            pin 3: push button - push a button attached to pin 3 to toggle
 *                                 the state of the switch.
 *            pin 4: proximity - uses an infra red proximity detector to toggle
 *                               the switch when something comes within 20 cm 
 *                               (proximity sense pin attached to analog 0)
 *            pin 5: light - uses a photoresistor to turn the switch on when
 *                           there is not much light and off when there is
 *                           (photo resistor attached to analog 1)
 *           
 * Wiring: Servo Switch Signal - pin 9
 *         Push Button = pulls pin 3 low
 *         Proximity Sensor = Analog 0
 *         Photo Resistor = Analog 1
 *
 * License: This work is licenced under the Creative Commons 
 *          Attribution-Share Alike 3.0 Unported License. To 
 *          view a copy of this licence, visit 
 *          http://creativecommons.org/licenses/by-sa/3.0/ 
 *          or send a letter to Creative Commons, 171 Second 
 *          Street, Suite 300, San Francisco, California 94105, 
 *          USA.
 *         
*/ 


//--------------------------------------------------------------------------
// Servo Switch - (SESW) Preamble START

#include <Servo.h> 
 
#define SERVOSWITCHONEPIN 9   //The pin the first servoSwitch is attached to
Servo servoSwitchOne;         //the Servo object connected to our light switch


//Switch Constants (can be adjusted to fine tune for your switch)
#define SWITCHTIME 500        //the time in milliseconds it takes to flick the switch

boolean switchState;          //the state if the switch
#define ON true               //true when on
#define OFF false             //false when off

#define ONDEGREES  180        //the number of degrees on the servo corresponding to ON
#define NEUTRALDEGREES  75    //the number of degrees on the servo corresponding to NEUTRAL
#define OFFDEGREES  0    //the number of degrees on the servo corresponding to OFF

// Servo Switch - (SESW) Preamble END
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
// Servo Switch - (SESW) Demo Mode Preamble START

//Demo Mode One Preamble - BUTTON
#define MODEPINBUTTON 3    //pull pin 3 low to go into button mode
#define BUTTONPIN 2        //the pin the button is attached to (active low)


//Demo Mode Two Preamble - PROXIMITY
#define MODEPINPROXIMITY 4 //pull pin 4 low to go to proximity mode
#define PROXIMITYPIN 0     //proximity sense pin goes to analog 0
#define PROXIMITYTHRESHOLD 400 //the value which intiates a toggle (about 20cm) when higher than within 20cm

//Demo Mode Three Preamble - LIGHT
#define MODEPINLIGHT 5 //pull pin 5 low to go to light mode
#define LIGHTPIN 1     //light sense pin goes to analog 1
#define LIGHTTHRESHOLD 305 //the value which intiates a toggle (when shined by a flashlight) ~1volt

//Demo Mode Four Preamble - Photo Gate
#define MODEPINGATE 6 //pull pin 5 low to go to light mode
#define GATEPIN 2     //gate sense pin goes to analog 2
#define GATETHRESHOLD 800 //the value which intiates a toggle (value drops to below 4 volts when beam is interupted)

// Servo Switch - (SESW) Demo ModePreamble END
//--------------------------------------------------------------------------

long lastRun;
                
 
void setup() 
{ 
  lastRun = millis();
  Serial.begin(9600);
  servoSwitchSetup();
  servoSwitchDemoModeSetup();
} 
 
 
void loop() 
{ 
  
  boolean options = checkOptions();

  if(!options){  //if no demo modes are selected
     if(millis() - lastRun > 5000){   //seeing if five seconds have elapsed since we last toggled
     toggle();      //turn the switch on 
     lastRun = millis();              //set lastRun to current time
     }

  }
  
  


} 

boolean checkOptions(){
 boolean returnValue = false;
 
 //Demo Mode 1 - BUTTON
 if(digitalRead(MODEPINBUTTON) == LOW){
   Serial.println("BUTTON");
  if(digitalRead(BUTTONPIN) == LOW){ toggle();delay(500); lastRun = millis();} //set lastRun to current time
  returnValue = true;
 }
 //Demo Mode 2 - PROXIMITY
 if(digitalRead(MODEPINPROXIMITY) == LOW){
   Serial.println("PROXIMITY");
  if(analogRead(PROXIMITYPIN) > PROXIMITYTHRESHOLD){ toggle();delay(1000); lastRun = millis();} //set lastRun to current time
  returnValue = true;
 }
//Demo Mode 3 - LIGHT
 if(digitalRead(MODEPINLIGHT) == LOW){
   Serial.println("LIGHT");
  if(analogRead(LIGHTPIN) < LIGHTTHRESHOLD){ toggle();delay(1000); lastRun = millis();} //set lastRun to current time
  returnValue = true;
 }
//Demo Mode 4 - PHOTOGATE
 if(digitalRead(MODEPINGATE) == LOW){
   Serial.println("GATE");
  if(analogRead(GATEPIN) > GATETHRESHOLD){ toggle();delay(2000);lastRun = millis();} //set lastRun to current time
  returnValue = true;
 }



 return returnValue;
}

//--------------------------------------------------------------------------
// Servo Switch - (SESW) Routines START

/* Sets up our servo switch by attaching the servo to the selected pin (must be 9 or 10) */
void servoSwitchSetup(){
  servoSwitchOne.attach(SERVOSWITCHONEPIN);  //Attaches the servo switch's servo 
  turnOn();
}

/* Flicks the switch on then returns to the neutral position */
void turnOn(){
  switchState = ON;
  servoSwitchOne.write(ONDEGREES);  //turns the servo to on degrees    
  delay(SWITCHTIME);           //waits for the time it takes to fully flick the switch
  goNeutral();                 //returns the servo to the neutral position
} 

void turnOff(){
  switchState = OFF;
  servoSwitchOne.write(OFFDEGREES);  //turns the servo to off degrees    
 delay(SWITCHTIME);           //waits for the time it takes to fully flick the switch
 goNeutral();                 //returns the servo to the neutral position
} 

void goNeutral(){
 servoSwitchOne.write(NEUTRALDEGREES); //moves the servo to the neutral position
}

void toggle(){
  if(switchState == ON){turnOff();}
  else{turnOn();}
}

// Servo Switch - (SESW) Routines END
//--------------------------------------------------------------------------


//--------------------------------------------------------------------------
// Servo Switch - (SESW) Demo Mode Routines START

void servoSwitchDemoModeSetup(){
//Demo Mode One Setup - BUTTON
  pinMode(MODEPINBUTTON, INPUT);
  digitalWrite(MODEPINBUTTON, HIGH);
  pinMode(BUTTONPIN, INPUT);
  digitalWrite(BUTTONPIN, HIGH);
//Demo Mode Two Setup - PROXIMITY
  pinMode(MODEPINPROXIMITY, INPUT);
  digitalWrite(MODEPINPROXIMITY, HIGH);
//Demo Mode Three Setup - LIGHT
  pinMode(MODEPINLIGHT, INPUT);
  digitalWrite(MODEPINLIGHT, HIGH);  
//Demo Mode Four Setup - PHOTOGATE
  pinMode(MODEPINGATE, INPUT);
  digitalWrite(MODEPINGATE, HIGH);  
}

// Servo Switch - (SESW) Demo Mode Routines END
//--------------------------------------------------------------------------

